<?php
// Start session
session_start();

// Check if user is logged in
if (!isset($_SESSION["user_id"]) || $_SESSION["user_type"] !== "applicant") {
    echo json_encode(["success" => false, "message" => "Unauthorized access"]);
    exit();
}

// Include database connection
require_once "../config/connection.php";

// Get applicant ID from session
$applicant_id = $_SESSION["user_id"];

// Check if form data is provided
if (!isset($_POST["job_id"]) || !isset($_POST["cover_letter"]) || !isset($_POST["expected_salary"]) || !isset($_FILES["cv_file"])) {
    echo json_encode(["success" => false, "message" => "Data tidak lengkap"]);
    exit();
}

// Get form data
$job_id = intval($_POST["job_id"]);
$cover_letter = $_POST["cover_letter"];
$expected_salary = str_replace(",", "", $_POST["expected_salary"]);

// Validate expected salary
if (!is_numeric($expected_salary)) {
    echo json_encode(["success" => false, "message" => "Gaji yang diharapkan harus berupa angka"]);
    exit();
}

// Check if CV file is uploaded
if ($_FILES["cv_file"]["error"] !== UPLOAD_ERR_OK) {
    echo json_encode(["success" => false, "message" => "Gagal mengunggah CV"]);
    exit();
}

// Check file type
$allowed_types = ["application/pdf", "application/msword", "application/vnd.openxmlformats-officedocument.wordprocessingml.document"];
if (!in_array($_FILES["cv_file"]["type"], $allowed_types)) {
    echo json_encode(["success" => false, "message" => "File harus berformat PDF, DOC, atau DOCX"]);
    exit();
}

// Check file size (max 2MB)
if ($_FILES["cv_file"]["size"] > 2 * 1024 * 1024) {
    echo json_encode(["success" => false, "message" => "Ukuran file maksimal 2MB"]);
    exit();
}

// Start transaction
$conn->begin_transaction();

try {
    // Check if applicant has already applied for this job
    $check_query = "SELECT * FROM applications WHERE applicant_id = ? AND job_id = ?";
    $check_stmt = $conn->prepare($check_query);
    $check_stmt->bind_param("ii", $applicant_id, $job_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();

    if ($check_result->num_rows > 0) {
        echo json_encode(["success" => false, "message" => "Anda sudah melamar posisi ini sebelumnya"]);
        exit();
    }

    // Insert application data
    $insert_query = "INSERT INTO applications (applicant_id, job_id, cover_letter, expected_salary) VALUES (?, ?, ?, ?)";
    $insert_stmt = $conn->prepare($insert_query);
    $insert_stmt->bind_param("iiss", $applicant_id, $job_id, $cover_letter, $expected_salary);
    $insert_stmt->execute();
    $application_id = $conn->insert_id;

    // Upload CV file
    $file_name = "CV_" . $applicant_id . "_" . $job_id . "_" . time();
    $file_extension = pathinfo($_FILES["cv_file"]["name"], PATHINFO_EXTENSION);
    $file_path = "../uploads/cv/" . $file_name . "." . $file_extension;
    $file_size = $_FILES["cv_file"]["size"];

    // Create directory if it doesn't exist
    if (!file_exists("../uploads/cv")) {
        mkdir("../uploads/cv", 0755, true);
    }

    // Move uploaded file
    if (!move_uploaded_file($_FILES["cv_file"]["tmp_name"], $file_path)) {
        throw new Exception("Failed to upload file");
    }

    // Insert document data
    $doc_query = "INSERT INTO documents (applicant_id, application_id, document_type, document_name, file_path, file_size, file_extension) VALUES (?, ?, 'cv', ?, ?, ?, ?)";
    $doc_stmt = $conn->prepare($doc_query);
    $doc_name = "CV " . $_FILES["cv_file"]["name"];
    $doc_stmt->bind_param("iissis", $applicant_id, $application_id, $doc_name, $file_path, $file_size, $file_extension);
    $doc_stmt->execute();

    // Create notification for applicant
    $notification_query = "INSERT INTO notifications (recipient_type, recipient_id, sender_type, sender_id, title, message, notification_type, related_id) VALUES ('applicant', ?, 'system', NULL, 'Lamaran Terkirim', ?, 'application_status', ?)";
    $notification_stmt = $conn->prepare($notification_query);
    
    // Get job title
    $job_query = "SELECT title FROM job_positions WHERE job_id = ?";
    $job_stmt = $conn->prepare($job_query);
    $job_stmt->bind_param("i", $job_id);
    $job_stmt->execute();
    $job_result = $job_stmt->get_result();
    $job = $job_result->fetch_assoc();
    $job_title = $job["title"];
    
    $message = "Lamaran Anda untuk posisi " . $job_title . " telah berhasil dikirim.";
    $notification_stmt->bind_param("isi", $applicant_id, $message, $application_id);
    $notification_stmt->execute();

    // Create notification for HR admin
    $hr_notification_query = "INSERT INTO notifications (recipient_type, recipient_id, sender_type, sender_id, title, message, notification_type, related_id) VALUES ('admin_recruitment', 1, 'applicant', ?, 'Lamaran Baru', ?, 'application_status', ?)";
    $hr_notification_stmt = $conn->prepare($hr_notification_query);
    
    // Get applicant name
    $applicant_query = "SELECT fullname FROM applicants WHERE id = ?";
    $applicant_stmt = $conn->prepare($applicant_query);
    $applicant_stmt->bind_param("i", $applicant_id);
    $applicant_stmt->execute();
    $applicant_result = $applicant_stmt->get_result();
    $applicant = $applicant_result->fetch_assoc();
    $applicant_name = $applicant["fullname"];
    
    $hr_message = "Lamaran baru untuk posisi " . $job_title . " dari " . $applicant_name . ".";
    $hr_notification_stmt->bind_param("isi", $applicant_id, $hr_message, $application_id);
    $hr_notification_stmt->execute();

    // Log the action
    $log_query = "INSERT INTO system_logs (user_type, user_id, action, table_affected, record_id, description, ip_address) VALUES ('applicant', ?, 'submit_application', 'applications', ?, ?, ?)";
    $log_stmt = $conn->prepare($log_query);
    $description = "Applicant submitted application for job ID " . $job_id;
    $ip_address = $_SERVER["REMOTE_ADDR"];
    $log_stmt->bind_param("iiss", $applicant_id, $application_id, $description, $ip_address);
    $log_stmt->execute();

    // Commit transaction
    $conn->commit();

    echo json_encode(["success" => true, "message" => "Lamaran berhasil dikirim"]);
} catch (Exception $e) {
    // Rollback transaction
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "Terjadi kesalahan: " . $e->getMessage()]);
}
?>