<?php
// Start session and check if user is logged in
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'applicant') {
    header("Location: ../login.php");
    exit();
}

// Include database connection
require_once '../config/connection.php';

// Get applicant data from database
$user_id = $_SESSION['user_id'];
$query = "SELECT * FROM applicants WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$applicant = $result->fetch_assoc();

// Get all active job positions
$query_jobs = "SELECT * FROM job_positions WHERE is_active = 1 ORDER BY created_at DESC";
$result_jobs = $conn->query($query_jobs);

// Include header
include '../template_applicants/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-soft-green via-white to-medium-green">
    <!-- Page Header -->
    <div class="flex justify-between items-center mb-6">
        <h2 class="text-2xl font-bold text-green-800">
            <i class="fas fa-briefcase mr-2"></i> Posisi Pekerjaan Tersedia
        </h2>
        <div class="flex gap-2">
            <div class="relative">
                <input type="text" id="searchJobs" placeholder="Cari posisi..." class="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 w-full">
                <i class="fas fa-search absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
            </div>
            <select id="filterLocation" class="border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-green-500 focus:border-green-500">
                <option value="">Semua Lokasi</option>
                <option value="surabaya">Surabaya</option>
                <option value="semarang">Semarang</option>
            </select>
            <select id="filterDepartment" class="border border-gray-300 rounded-lg px-4 py-2 focus:ring-2 focus:ring-green-500 focus:border-green-500">
                <option value="">Semua Departemen</option>
                <option value="IT">IT</option>
                <option value="Produksi">Produksi</option>
            </select>
        </div>
    </div>
    
    <!-- Jobs Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6" id="jobsContainer">
        <?php if ($result_jobs->num_rows > 0): ?>
            <?php while ($job = $result_jobs->fetch_assoc()): ?>
                <div class="job-card bg-white rounded-lg shadow-md hover:shadow-lg transition-all duration-300 overflow-hidden border border-gray-200 flex flex-col" 
                     data-location="<?php echo htmlspecialchars($job['lokasi_penempatan']); ?>"
                     data-department="<?php echo htmlspecialchars($job['department']); ?>">
                    <!-- Card Header with Tag -->
                    <div class="flex justify-between items-center p-4 border-b border-gray-100 bg-soft-green">
                        <h3 class="font-semibold text-green-800 text-lg"><?php echo htmlspecialchars($job['title']); ?></h3>
                        <span class="px-2 py-1 text-xs font-medium rounded-full <?php echo $job['lokasi_penempatan'] == 'surabaya' ? 'bg-blue-100 text-blue-800' : 'bg-purple-100 text-purple-800'; ?>">
                            <?php echo ucfirst(htmlspecialchars($job['lokasi_penempatan'])); ?>
                        </span>
                    </div>
                    
                    <!-- Card Body -->
                    <div class="p-4 flex-grow">
                        <div class="flex items-center mb-3 text-sm text-gray-600">
                            <i class="fas fa-building mr-2"></i>
                            <span><?php echo htmlspecialchars($job['department']); ?></span>
                        </div>
                        <div class="flex items-center mb-3 text-sm text-gray-600">
                            <i class="fas fa-map-marker-alt mr-2"></i>
                            <span><?php echo htmlspecialchars($job['location']); ?></span>
                        </div>
                        <div class="flex items-center mb-3 text-sm text-gray-600">
                            <i class="fas fa-clock mr-2"></i>
                            <span><?php echo htmlspecialchars($job['job_type']); ?></span>
                        </div>
                        <div class="flex items-center mb-3 text-sm text-gray-600">
                            <i class="fas fa-money-bill-wave mr-2"></i>
                            <span><?php echo htmlspecialchars($job['salary_range']); ?></span>
                        </div>
                        <div class="flex items-center mb-3 text-sm text-gray-600">
                            <i class="fas fa-user-friends mr-2"></i>
                            <span>Dibutuhkan: <?php echo htmlspecialchars($job['vacancy_count']); ?> orang</span>
                        </div>
                        <p class="text-sm text-gray-600 mb-4 line-clamp-2">
                            <?php echo htmlspecialchars($job['description']); ?>
                        </p>
                    </div>
                    
                    <!-- Card Footer with Action Buttons -->
                    <div class="p-4 border-t border-gray-100 flex gap-2">
                        <button class="btn-details flex-1 bg-gray-100 hover:bg-gray-200 text-gray-800 py-2 px-4 rounded-md transition-colors duration-300 text-sm flex items-center justify-center" 
                                data-job-id="<?php echo $job['job_id']; ?>">
                            <i class="fas fa-info-circle mr-1"></i> Detail
                        </button>
                        <button class="btn-apply flex-1 bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-md transition-colors duration-300 text-sm flex items-center justify-center"
                                data-job-id="<?php echo $job['job_id']; ?>"
                                data-job-title="<?php echo htmlspecialchars($job['title']); ?>"
                                data-job-department="<?php echo htmlspecialchars($job['department']); ?>"
                                data-job-location="<?php echo htmlspecialchars($job['lokasi_penempatan']); ?>">
                            <i class="fas fa-paper-plane mr-1"></i> Lamar
                        </button>
                    </div>
                </div>
            <?php endwhile; ?>
        <?php else: ?>
            <div class="col-span-full flex flex-col items-center justify-center p-8 bg-white rounded-lg shadow">
                <img src="../images/empty-state.svg" alt="No jobs available" class="w-64 mb-4">
                <h3 class="text-xl font-medium text-gray-800">Belum ada posisi pekerjaan tersedia</h3>
                <p class="text-gray-600">Silakan cek kembali nanti untuk informasi lowongan terbaru.</p>
            </div>
        <?php endif; ?>
    </div>
</main>

<!-- Job Details Modal -->
<div id="jobDetailsModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-lg shadow-lg w-full max-w-3xl max-h-[90vh] overflow-hidden flex flex-col">
        <div class="bg-green-700 px-6 py-4 flex justify-between items-center">
            <h3 class="text-xl font-bold text-white" id="modalJobTitle">Detail Posisi</h3>
            <button class="text-white hover:text-gray-200 close-modal">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="p-6 overflow-y-auto flex-grow" id="jobDetailsContent">
            <!-- Content will be loaded dynamically -->
            <div class="animate-pulse">
                <div class="h-4 bg-gray-200 rounded w-3/4 mb-4"></div>
                <div class="h-4 bg-gray-200 rounded w-1/2 mb-6"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-3/4 mb-6"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                <div class="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
            </div>
        </div>
        <div class="px-6 py-4 bg-gray-50 border-t border-gray-200">
            <button class="bg-green-600 hover:bg-green-700 text-white py-2 px-6 rounded-md transition-colors duration-300 close-modal">
                Tutup
            </button>
        </div>
    </div>
</div>

<!-- Apply Job Modal -->
<div id="applyJobModal" class="fixed inset-0 bg-gray-900 bg-opacity-50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-lg shadow-lg w-full max-w-2xl max-h-[90vh] overflow-hidden flex flex-col">
        <div class="bg-green-700 px-6 py-4 flex justify-between items-center">
            <h3 class="text-xl font-bold text-white">Formulir Lamaran</h3>
            <button class="text-white hover:text-gray-200 close-modal">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <form id="applicationForm" enctype="multipart/form-data" class="p-6 overflow-y-auto flex-grow">
            <input type="hidden" id="job_id" name="job_id">
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2">Posisi:</label>
                <input type="text" id="job_title" class="w-full p-2 border border-gray-300 rounded-md bg-gray-100" readonly>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2">Departemen:</label>
                <input type="text" id="job_department" class="w-full p-2 border border-gray-300 rounded-md bg-gray-100" readonly>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2">Lokasi Penempatan:</label>
                <input type="text" id="job_location" class="w-full p-2 border border-gray-300 rounded-md bg-gray-100" readonly>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2" for="cover_letter">Surat Lamaran/Cover Letter:</label>
                <textarea id="cover_letter" name="cover_letter" rows="5" class="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-green-500 focus:border-green-500" placeholder="Tulis surat lamaran Anda di sini..."></textarea>
            </div>

            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2" for="expected_salary">Gaji yang Diharapkan (Rp):</label>
                <input type="text" id="expected_salary" name="expected_salary" class="w-full p-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-green-500 focus:border-green-500" placeholder="Contoh: 6000000">
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-medium mb-2" for="cv_file">Unggah CV (PDF, DOC, DOCX):</label>
                <div class="border-2 border-dashed border-gray-300 rounded-md p-4 hover:border-green-500 transition-colors">
                    <div class="flex flex-col items-center justify-center">
                        <i class="fas fa-file-upload text-3xl text-gray-400 mb-2"></i>
                        <p class="text-sm text-gray-500 mb-2">Klik atau seret file ke sini</p>
                        <p class="text-xs text-gray-400">Maksimal 2MB</p>
                        <input type="file" id="cv_file" name="cv_file" accept=".pdf,.doc,.docx" class="hidden">
                    </div>
                </div>
                <div id="file_info" class="mt-2 text-sm hidden">
                    <span class="text-gray-700">File terpilih: </span>
                    <span id="file_name" class="font-medium text-green-600"></span>
                </div>
            </div>
            
            <div id="application_response" class="mb-4 hidden">
                <!-- Response message will appear here -->
            </div>
        </form>
        <div class="px-6 py-4 bg-gray-50 border-t border-gray-200 flex justify-between">
            <button class="bg-gray-500 hover:bg-gray-600 text-white py-2 px-6 rounded-md transition-colors duration-300 close-modal">
                Batal
            </button>
            <button type="submit" form="applicationForm" class="bg-green-600 hover:bg-green-700 text-white py-2 px-6 rounded-md transition-colors duration-300">
                <i class="fas fa-paper-plane mr-1"></i> Kirim Lamaran
            </button>
        </div>
    </div>
</div>

<!-- Script for Job Card and Modal Functionality -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Get DOM Elements
    const jobDetailsModal = document.getElementById('jobDetailsModal');
    const jobDetailsContent = document.getElementById('jobDetailsContent');
    const modalJobTitle = document.getElementById('modalJobTitle');
    const applyJobModal = document.getElementById('applyJobModal');
    const applicationForm = document.getElementById('applicationForm');
    const jobIdInput = document.getElementById('job_id');
    const jobTitleInput = document.getElementById('job_title');
    const jobDepartmentInput = document.getElementById('job_department');
    const jobLocationInput = document.getElementById('job_location');
    const fileInput = document.getElementById('cv_file');
    const fileInfo = document.getElementById('file_info');
    const fileName = document.getElementById('file_name');
    const responseMessage = document.getElementById('application_response');
    const jobsContainer = document.getElementById('jobsContainer');
    const searchInput = document.getElementById('searchJobs');
    const filterLocation = document.getElementById('filterLocation');
    const filterDepartment = document.getElementById('filterDepartment');

    // Handle click on Details button
    document.querySelectorAll('.btn-details').forEach(button => {
        button.addEventListener('click', function() {
            const jobId = this.getAttribute('data-job-id');
            modalJobTitle.textContent = 'Memuat Detail Pekerjaan...';
            jobDetailsContent.innerHTML = `
                <div class="animate-pulse">
                    <div class="h-4 bg-gray-200 rounded w-3/4 mb-4"></div>
                    <div class="h-4 bg-gray-200 rounded w-1/2 mb-6"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-3/4 mb-6"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-full mb-2"></div>
                    <div class="h-4 bg-gray-200 rounded w-3/4 mb-2"></div>
                </div>
            `;
            
            // Show modal
            jobDetailsModal.classList.remove('hidden');
            jobDetailsModal.classList.add('flex');
            
            // Fetch job details using AJAX
            fetch(`get_job_details.php?job_id=${jobId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        modalJobTitle.textContent = data.job.title;
                        
                        // Format the job details
                        const requirements = data.job.requirements.split('\n').map(req => `<li class="mb-2">${req}</li>`).join('');
                        const responsibilities = data.job.responsibilities.split('\n').map(resp => `<li class="mb-2">${resp}</li>`).join('');
                        
                        jobDetailsContent.innerHTML = `
                            <div class="mb-6">
                                <div class="flex flex-wrap gap-2 mb-4">
                                    <span class="px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm font-medium">${data.job.department}</span>
                                    <span class="px-3 py-1 bg-green-100 text-green-800 rounded-full text-sm font-medium">${data.job.job_type}</span>
                                    <span class="px-3 py-1 bg-purple-100 text-purple-800 rounded-full text-sm font-medium">${data.job.lokasi_penempatan}</span>
                                </div>
                                <p class="text-gray-700">${data.job.description}</p>
                            </div>
                            
                            <div class="mb-6">
                                <h4 class="text-lg font-semibold text-green-800 mb-3">Detail Pekerjaan</h4>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 bg-gray-50 p-4 rounded-lg">
                                    <div class="flex items-start">
                                        <div class="text-green-500 mr-3"><i class="fas fa-map-marker-alt"></i></div>
                                        <div>
                                            <h5 class="font-medium text-gray-700">Lokasi</h5>
                                            <p class="text-gray-600">${data.job.location}</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start">
                                        <div class="text-green-500 mr-3"><i class="fas fa-money-bill-wave"></i></div>
                                        <div>
                                            <h5 class="font-medium text-gray-700">Kisaran Gaji</h5>
                                            <p class="text-gray-600">${data.job.salary_range}</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start">
                                        <div class="text-green-500 mr-3"><i class="fas fa-user-friends"></i></div>
                                        <div>
                                            <h5 class="font-medium text-gray-700">Jumlah Lowongan</h5>
                                            <p class="text-gray-600">${data.job.vacancy_count} orang</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start">
                                        <div class="text-green-500 mr-3"><i class="fas fa-calendar-alt"></i></div>
                                        <div>
                                            <h5 class="font-medium text-gray-700">Tanggal Posting</h5>
                                            <p class="text-gray-600">${new Date(data.job.created_at).toLocaleDateString('id-ID', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-6">
                                <h4 class="text-lg font-semibold text-green-800 mb-3">Persyaratan</h4>
                                <ul class="list-disc pl-5 text-gray-700">
                                    ${requirements}
                                </ul>
                            </div>
                            
                            <div class="mb-6">
                                <h4 class="text-lg font-semibold text-green-800 mb-3">Tanggung Jawab</h4>
                                <ul class="list-disc pl-5 text-gray-700">
                                    ${responsibilities}
                                </ul>
                            </div>
                        `;
                    } else {
                        jobDetailsContent.innerHTML = `
                            <div class="text-center p-6">
                                <i class="fas fa-exclamation-circle text-red-500 text-4xl mb-4"></i>
                                <p class="text-gray-700">Terjadi kesalahan saat memuat detail pekerjaan. Silakan coba lagi nanti.</p>
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    jobDetailsContent.innerHTML = `
                        <div class="text-center p-6">
                            <i class="fas fa-exclamation-circle text-red-500 text-4xl mb-4"></i>
                            <p class="text-gray-700">Terjadi kesalahan saat memuat detail pekerjaan. Silakan coba lagi nanti.</p>
                        </div>
                    `;
                    console.error('Error fetching job details:', error);
                });
        });
    });

    // Handle click on Apply button
    document.querySelectorAll('.btn-apply').forEach(button => {
        button.addEventListener('click', function() {
            const jobId = this.getAttribute('data-job-id');
            const jobTitle = this.getAttribute('data-job-title');
            const jobDepartment = this.getAttribute('data-job-department');
            const jobLocation = this.getAttribute('data-job-location');
            
            // Fill form fields
            jobIdInput.value = jobId;
            jobTitleInput.value = jobTitle;
            jobDepartmentInput.value = jobDepartment;
            jobLocationInput.value = jobLocation.charAt(0).toUpperCase() + jobLocation.slice(1);
            
            // Reset form and response
            applicationForm.reset();
            fileInfo.classList.add('hidden');
            responseMessage.classList.add('hidden');
            
            // Show modal
            applyJobModal.classList.remove('hidden');
            applyJobModal.classList.add('flex');
        });
    });

    // Handle file selection
    const dropArea = document.querySelector('.border-dashed');
    
    dropArea.addEventListener('click', function() {
        fileInput.click();
    });
    
    fileInput.addEventListener('change', function() {
        if (this.files && this.files[0]) {
            const file = this.files[0];
            fileName.textContent = file.name;
            fileInfo.classList.remove('hidden');
            
            // Validate file type and size
            const validTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
            const maxSize = 2 * 1024 * 1024; // 2MB
            
            if (!validTypes.includes(file.type)) {
                responseMessage.innerHTML = `
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                        <strong class="font-bold">Error!</strong>
                        <span class="block sm:inline"> File harus berformat PDF, DOC, atau DOCX.</span>
                    </div>
                `;
                responseMessage.classList.remove('hidden');
                fileInput.value = ''; // Clear the file input
                fileInfo.classList.add('hidden');
                return;
            }
            
            if (file.size > maxSize) {
                responseMessage.innerHTML = `
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                        <strong class="font-bold">Error!</strong>
                        <span class="block sm:inline"> Ukuran file maksimal 2MB.</span>
                    </div>
                `;
                responseMessage.classList.remove('hidden');
                fileInput.value = ''; // Clear the file input
                fileInfo.classList.add('hidden');
                return;
            }
        }
    });
    
    // Handle drag and drop
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, (e) => {
            e.preventDefault();
            e.stopPropagation();
        }, false);
    });
    
    ['dragenter', 'dragover'].forEach(eventName => {
        dropArea.addEventListener(eventName, () => {
            dropArea.classList.add('border-green-500', 'bg-green-50');
        }, false);
    });
    
    ['dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, () => {
            dropArea.classList.remove('border-green-500', 'bg-green-50');
        }, false);
    });
    
    dropArea.addEventListener('drop', (e) => {
        fileInput.files = e.dataTransfer.files;
        const event = new Event('change');
        fileInput.dispatchEvent(event);
    }, false);

    // Handle form submission
    applicationForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Show loading state
        document.querySelector('[type=submit]').innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Mengirim...';
        document.querySelector('[type=submit]').disabled = true;
        
        // Create form data object
        const formData = new FormData(this);
        
        // Send AJAX request
        fetch('submit_application.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            // Reset button state
            document.querySelector('[type=submit]').innerHTML = '<i class="fas fa-paper-plane mr-1"></i> Kirim Lamaran';
            document.querySelector('[type=submit]').disabled = false;
            
            if (data.success) {
                responseMessage.innerHTML = `
                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                        <strong class="font-bold">Sukses!</strong>
                        <span class="block sm:inline"> ${data.message}</span>
                    </div>
                `;
                responseMessage.classList.remove('hidden');
                
                // Reset form after successful submission
                applicationForm.reset();
                fileInfo.classList.add('hidden');
                
                // Close modal after a delay
                setTimeout(() => {
                    applyJobModal.classList.add('hidden');
                    applyJobModal.classList.remove('flex');
                    
                    // Show success toast
                    showToast('Lamaran berhasil dikirim!', 'success');
                }, 2000);
            } else {
                responseMessage.innerHTML = `
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                        <strong class="font-bold">Error!</strong>
                        <span class="block sm:inline"> ${data.message}</span>
                    </div>
                `;
                responseMessage.classList.remove('hidden');
            }
        })
        .catch(error => {
            // Reset button state
            document.querySelector('[type=submit]').innerHTML = '<i class="fas fa-paper-plane mr-1"></i> Kirim Lamaran';
            document.querySelector('[type=submit]').disabled = false;
            
            responseMessage.innerHTML = `
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded">
                    <strong class="font-bold">Error!</strong>
                    <span class="block sm:inline"> Terjadi kesalahan saat mengirim lamaran. Silakan coba lagi nanti.</span>
                </div>
            `;
            responseMessage.classList.remove('hidden');
            console.error('Error submitting application:', error);
        });
    });

    // Close modals when clicking the close button or outside
    document.querySelectorAll('.close-modal').forEach(button => {
        button.addEventListener('click', function() {
            jobDetailsModal.classList.add('hidden');
            jobDetailsModal.classList.remove('flex');
            applyJobModal.classList.add('hidden');
            applyJobModal.classList.remove('flex');
        });
    });
    
    document.addEventListener('click', function(e) {
        if (e.target === jobDetailsModal) {
            jobDetailsModal.classList.add('hidden');
            jobDetailsModal.classList.remove('flex');
        }
        if (e.target === applyJobModal) {
            applyJobModal.classList.add('hidden');
            applyJobModal.classList.remove('flex');
        }
    });

    // Search and filter functionality
    function applyFilters() {
        const searchValue = searchInput.value.toLowerCase();
        const locationValue = filterLocation.value;
        const departmentValue = filterDepartment.value;
        
        document.querySelectorAll('.job-card').forEach(card => {
            const title = card.querySelector('h3').textContent.toLowerCase();
            const location = card.getAttribute('data-location');
            const department = card.getAttribute('data-department');
            
            const matchesSearch = title.includes(searchValue);
            const matchesLocation = locationValue === '' || location === locationValue;
            const matchesDepartment = departmentValue === '' || department === departmentValue;
            
            if (matchesSearch && matchesLocation && matchesDepartment) {
                card.style.display = 'flex';
            } else {
                card.style.display = 'none';
            }
        });
        
        // Show empty state if no jobs match filters
        const visibleJobs = document.querySelectorAll('.job-card[style="display: flex"]').length;
        const emptyState = document.querySelector('.col-span-full');
        
        if (visibleJobs === 0 && !emptyState) {
            const emptyStateElement = document.createElement('div');
            emptyStateElement.className = 'col-span-full flex flex-col items-center justify-center p-8 bg-white rounded-lg shadow';
            emptyStateElement.innerHTML = `
                <img src="../images/empty-filter.svg" alt="No matching jobs" class="w-64 mb-4">
                <h3 class="text-xl font-medium text-gray-800">Tidak ada posisi yang sesuai filter</h3>
                <p class="text-gray-600">Coba ubah filter pencarian Anda.</p>
                <button id="resetFilters" class="mt-4 bg-green-600 hover:bg-green-700 text-white py-2 px-4 rounded-md transition-colors duration-300">
                    Reset Filter
                </button>
            `;
            jobsContainer.appendChild(emptyStateElement);
            
            // Add event listener to reset filters button
            document.getElementById('resetFilters').addEventListener('click', function() {
                searchInput.value = '';
                filterLocation.value = '';
                filterDepartment.value = '';
                applyFilters();
            });
        } else if (visibleJobs > 0 && emptyState) {
            emptyState.remove();
        }
    }
    
    searchInput.addEventListener('input', applyFilters);
    filterLocation.addEventListener('change', applyFilters);
    filterDepartment.addEventListener('change', applyFilters);
});
</script>

<?php
// Create PHP file handler for job details
$handler_file = '../api/get_job_details.php';
$handler_content = '<?php
// Include database connection
require_once "../config/connection.php";

// Check if job ID is provided
if (!isset($_GET["job_id"])) {
    echo json_encode(["success" => false, "message" => "Job ID not provided"]);
    exit();
}

$job_id = intval($_GET["job_id"]);

// Get job details from database
$query = "SELECT * FROM job_positions WHERE job_id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $job_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    echo json_encode(["success" => false, "message" => "Job not found"]);
    exit();
}

$job = $result->fetch_assoc();

// Return job details as JSON
echo json_encode(["success" => true, "job" => $job]);
?>';

// Create the directory if it doesn't exist
if (!file_exists('../api')) {
    mkdir('../api', 0755, true);
}

// Write the handler file
file_put_contents($handler_file, $handler_content);

// Create PHP file handler for submitting application
$application_handler_file = '../api/submit_application.php';
$application_handler_content = '<?php
// Start session
session_start();

// Check if user is logged in
if (!isset($_SESSION["user_id"]) || $_SESSION["user_type"] !== "applicant") {
    echo json_encode(["success" => false, "message" => "Unauthorized access"]);
    exit();
}

// Include database connection
require_once "../config/connection.php";

// Get applicant ID from session
$applicant_id = $_SESSION["user_id"];

// Check if form data is provided
if (!isset($_POST["job_id"]) || !isset($_POST["cover_letter"]) || !isset($_POST["expected_salary"]) || !isset($_FILES["cv_file"])) {
    echo json_encode(["success" => false, "message" => "Data tidak lengkap"]);
    exit();
}

// Get form data
$job_id = intval($_POST["job_id"]);
$cover_letter = $_POST["cover_letter"];
$expected_salary = str_replace(",", "", $_POST["expected_salary"]);

// Validate expected salary
if (!is_numeric($expected_salary)) {
    echo json_encode(["success" => false, "message" => "Gaji yang diharapkan harus berupa angka"]);
    exit();
}

// Check if CV file is uploaded
if ($_FILES["cv_file"]["error"] !== UPLOAD_ERR_OK) {
    echo json_encode(["success" => false, "message" => "Gagal mengunggah CV"]);
    exit();
}

// Check file type
$allowed_types = ["application/pdf", "application/msword", "application/vnd.openxmlformats-officedocument.wordprocessingml.document"];
if (!in_array($_FILES["cv_file"]["type"], $allowed_types)) {
    echo json_encode(["success" => false, "message" => "File harus berformat PDF, DOC, atau DOCX"]);
    exit();
}

// Check file size (max 2MB)
if ($_FILES["cv_file"]["size"] > 2 * 1024 * 1024) {
    echo json_encode(["success" => false, "message" => "Ukuran file maksimal 2MB"]);
    exit();
}

// Start transaction
$conn->begin_transaction();

try {
    // Check if applicant has already applied for this job
    $check_query = "SELECT * FROM applications WHERE applicant_id = ? AND job_id = ?";
    $check_stmt = $conn->prepare($check_query);
    $check_stmt->bind_param("ii", $applicant_id, $job_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();

    if ($check_result->num_rows > 0) {
        echo json_encode(["success" => false, "message" => "Anda sudah melamar posisi ini sebelumnya"]);
        exit();
    }

    // Insert application data
    $insert_query = "INSERT INTO applications (applicant_id, job_id, cover_letter, expected_salary) VALUES (?, ?, ?, ?)";
    $insert_stmt = $conn->prepare($insert_query);
    $insert_stmt->bind_param("iiss", $applicant_id, $job_id, $cover_letter, $expected_salary);
    $insert_stmt->execute();
    $application_id = $conn->insert_id;

    // Upload CV file
    $file_name = "CV_" . $applicant_id . "_" . $job_id . "_" . time();
    $file_extension = pathinfo($_FILES["cv_file"]["name"], PATHINFO_EXTENSION);
    $file_path = "../uploads/cv/" . $file_name . "." . $file_extension;
    $file_size = $_FILES["cv_file"]["size"];

    // Create directory if it doesn\'t exist
    if (!file_exists("../uploads/cv")) {
        mkdir("../uploads/cv", 0755, true);
    }

    // Move uploaded file
    if (!move_uploaded_file($_FILES["cv_file"]["tmp_name"], $file_path)) {
        throw new Exception("Failed to upload file");
    }

    // Insert document data
    $doc_query = "INSERT INTO documents (applicant_id, application_id, document_type, document_name, file_path, file_size, file_extension) VALUES (?, ?, \'cv\', ?, ?, ?, ?)";
    $doc_stmt = $conn->prepare($doc_query);
    $doc_name = "CV " . $_FILES["cv_file"]["name"];
    $doc_stmt->bind_param("iissis", $applicant_id, $application_id, $doc_name, $file_path, $file_size, $file_extension);
    $doc_stmt->execute();

    // Create notification for applicant
    $notification_query = "INSERT INTO notifications (recipient_type, recipient_id, sender_type, sender_id, title, message, notification_type, related_id) VALUES (\'applicant\', ?, \'system\', NULL, \'Lamaran Terkirim\', \'Lamaran Anda untuk posisi {job_title} telah berhasil dikirim.\', \'application_status\', ?)";
    $notification_stmt = $conn->prepare($notification_query);
    
    // Get job title
    $job_query = "SELECT title FROM job_positions WHERE job_id = ?";
    $job_stmt = $conn->prepare($job_query);
    $job_stmt->bind_param("i", $job_id);
    $job_stmt->execute();
    $job_result = $job_stmt->get_result();
    $job = $job_result->fetch_assoc();
    $job_title = $job["title"];
    
    $message = str_replace("{job_title}", $job_title, "Lamaran Anda untuk posisi {job_title} telah berhasil dikirim.");
    $notification_stmt->bind_param("is", $applicant_id, $application_id, $message);
    $notification_stmt->execute();

    // Create notification for HR admin
    $hr_notification_query = "INSERT INTO notifications (recipient_type, recipient_id, sender_type, sender_id, title, message, notification_type, related_id) VALUES (\'admin_recruitment\', 1, \'applicant\', ?, \'Lamaran Baru\', \'Lamaran baru untuk posisi {job_title} dari {applicant_name}.\', \'application_status\', ?)";
    $hr_notification_stmt = $conn->prepare($hr_notification_query);
    
    // Get applicant name
    $applicant_query = "SELECT fullname FROM applicants WHERE id = ?";
    $applicant_stmt = $conn->prepare($applicant_query);
    $applicant_stmt->bind_param("i", $applicant_id);
    $applicant_stmt->execute();
    $applicant_result = $applicant_stmt->get_result();
    $applicant = $applicant_result->fetch_assoc();
    $applicant_name = $applicant["fullname"];
    
    $hr_message = "Lamaran baru untuk posisi " . $job_title . " dari " . $applicant_name . ".";
    $hr_notification_stmt->bind_param("ii", $applicant_id, $application_id);
    $hr_notification_stmt->execute();

    // Log the action
    $log_query = "INSERT INTO system_logs (user_type, user_id, action, table_affected, record_id, description, ip_address) VALUES (\'applicant\', ?, \'submit_application\', \'applications\', ?, ?, ?)";
    $log_stmt = $conn->prepare($log_query);
    $description = "Applicant submitted application for job ID " . $job_id;
    $ip_address = $_SERVER["REMOTE_ADDR"];
    $log_stmt->bind_param("iiss", $applicant_id, $application_id, $description, $ip_address);
    $log_stmt->execute();

    // Commit transaction
    $conn->commit();

    echo json_encode(["success" => true, "message" => "Lamaran berhasil dikirim"]);
} catch (Exception $e) {
    // Rollback transaction
    $conn->rollback();
    echo json_encode(["success" => false, "message" => "Terjadi kesalahan: " . $e->getMessage()]);
}
?>';

// Create the directory if it doesn't exist
if (!file_exists('../api')) {
    mkdir('../api', 0755, true);
}

// Write the application handler file
file_put_contents($application_handler_file, $application_handler_content);

include '../template_applicants/footer.php';
?>