<?php
// Start session and check if user is logged in
session_start();
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'admin_it') {
    header("Location: ../login.php");
    exit();
}

// Include database connection
require_once '../config/connection.php';

// Get admin data from database
$user_id = $_SESSION['user_id'];
$query = "SELECT * FROM admin_it WHERE id = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$admin = $result->fetch_assoc();

// Process delete request
if (isset($_POST['delete_job']) && isset($_POST['job_id'])) {
    $job_id = $_POST['job_id'];
    
    // Delete job position
    $delete_query = "DELETE FROM job_positions WHERE job_id = ? AND created_by = ?";
    $stmt = $conn->prepare($delete_query);
    $stmt->bind_param("ii", $job_id, $user_id);
    
    if ($stmt->execute()) {
        $success_message = "Posisi pekerjaan berhasil dihapus!";
    } else {
        $error_message = "Error: " . $stmt->error;
    }
    
    // If it's an AJAX request, return JSON response
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        $response = ['success' => empty($error_message), 'message' => !empty($success_message) ? $success_message : $error_message];
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
}

// Process update request
if (isset($_POST['update_job'])) {
    $job_id = $_POST['job_id'];
    $title = trim($_POST['title']);
    $department = trim($_POST['department']);
    $description = trim($_POST['description']);
    $requirements = trim($_POST['requirements']);
    $benefits = trim($_POST['benefits']);
    $information = trim($_POST['information']);
    $job_type = $_POST['job_type'];
    $location = trim($_POST['location']);
    $lokasi_penempatan = $_POST['lokasi_penempatan'];
    $salary_range = trim($_POST['salary_range']);
    $vacancy_count = (int)$_POST['vacancy_count'];
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    $link = trim($_POST['link']);
    
    // Basic validation
    if (empty($title) || empty($department) || empty($description) || 
        empty($requirements) || empty($benefits) || empty($location)) {
        $error_message = "Semua field wajib diisi!";
    } else {
        // Update job position
        $update_query = "UPDATE job_positions SET 
                          title = ?, 
                          department = ?, 
                          description = ?, 
                          requirements = ?, 
                          benefits = ?, 
                          information = ?,
                          job_type = ?, 
                          location = ?, 
                          lokasi_penempatan = ?, 
                          salary_range = ?, 
                          vacancy_count = ?, 
                          is_active = ?,
                          link = ?,
                          updated_at = CURRENT_TIMESTAMP,
                          updated_by = ?
                          WHERE job_id = ?";
        
        $stmt = $conn->prepare($update_query);
        $stmt->bind_param(
            "ssssssssssiisii", 
            $title, 
            $department, 
            $description, 
            $requirements, 
            $benefits, 
            $information,
            $job_type, 
            $location, 
            $lokasi_penempatan, 
            $salary_range, 
            $vacancy_count, 
            $is_active,
            $link,
            $user_id,
            $job_id
        );
        
        if ($stmt->execute()) {
            $success_message = "Posisi pekerjaan berhasil diperbarui!";
        } else {
            $error_message = "Error: " . $stmt->error;
        }
    }
    
    // If it's an AJAX request, return JSON response
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        // Get the updated job data
        $job_query = "SELECT * FROM job_positions WHERE job_id = ?";
        $stmt = $conn->prepare($job_query);
        $stmt->bind_param("i", $job_id);
        $stmt->execute();
        $updated_job = $stmt->get_result()->fetch_assoc();
        
        $response = [
            'success' => empty($error_message), 
            'message' => !empty($success_message) ? $success_message : $error_message,
            'job' => $updated_job
        ];
        
        header('Content-Type: application/json');
        echo json_encode($response);
        exit;
    }
}

// Get all job positions
$query = "SELECT * FROM job_positions ORDER BY created_at DESC";
$result = $conn->query($query);
$job_positions = [];

if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $job_positions[] = $row;
    }
}

// Include header
include '../template_it/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-soft-green via-white to-medium-green">
    <div class="max-w-7xl mx-auto">
        <!-- Page Header -->
        <div class="bg-white rounded-xl shadow-md overflow-hidden mb-6">
            <div class="bg-gradient-primary p-4 text-white flex justify-between items-center">
                <div>
                    <h1 class="text-2xl font-bold">Kelola Posisi Pekerjaan</h1>
                    <p class="text-sm opacity-80">Lihat, edit, dan hapus posisi pekerjaan yang telah dibuat</p>
                </div>
                <a href="buat_posisi_it.php" class="bg-white text-green-600 px-4 py-2 rounded-lg hover:bg-green-50 transition-colors duration-300 flex items-center shadow-sm">
                    <i class="fas fa-plus mr-2"></i> Buat Posisi Baru
                </a>
            </div>
            
            <!-- Search & Filter -->
            <div class="bg-gray-50 p-4 border-b">
                <div class="flex flex-col md:flex-row gap-4">
                    <div class="flex-1">
                        <div class="relative">
                            <input type="text" id="search-job" placeholder="Cari posisi pekerjaan..." class="w-full px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 pl-10">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                        </div>
                    </div>
                    <div class="flex gap-2">
                        <select id="filter-penempatan" class="px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 bg-white">
                            <option value="">Semua Lokasi</option>
                            <option value="surabaya">Surabaya</option>
                            <option value="semarang">Semarang</option>
                        </select>
                        <select id="filter-type" class="px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 bg-white">
                            <option value="">Semua Tipe</option>
                            <option value="full-time">Full Time</option>
                            <option value="part-time">Part Time</option>
                            <option value="contract">Kontrak</option>
                            <option value="internship">Magang</option>
                        </select>
                        <select id="filter-status" class="px-4 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500 bg-white">
                            <option value="">Semua Status</option>
                            <option value="1">Aktif</option>
                            <option value="0">Tidak Aktif</option>
                        </select>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Notification container -->
        <div id="notificationContainer" class="mb-4"></div>
        
        <!-- Job Positions Grid -->
        <div id="job-positions-grid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            <?php if (empty($job_positions)): ?>
                <div class="col-span-full bg-white p-8 rounded-xl shadow-md text-center">
                    <div class="text-gray-400 mb-4"><i class="fas fa-search text-5xl"></i></div>
                    <h3 class="text-xl font-medium text-gray-700 mb-2">Tidak ada posisi pekerjaan</h3>
                    <p class="text-gray-500 mb-4">Belum ada posisi pekerjaan yang dibuat. Klik tombol di bawah untuk membuat posisi baru.</p>
                    <a href="buat_posisi_it.php" class="inline-block bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors duration-300">
                        <i class="fas fa-plus mr-2"></i> Buat Posisi Baru
                    </a>
                </div>
            <?php else: ?>
                <?php foreach ($job_positions as $job): ?>
                <div id="job-card-<?php echo $job['job_id']; ?>" class="bg-white rounded-xl shadow-md overflow-hidden job-card transition-all duration-300 hover:shadow-lg" 
                     data-title="<?php echo strtolower($job['title']); ?>"
                     data-department="<?php echo strtolower($job['department']); ?>"
                     data-location="<?php echo strtolower($job['location']); ?>"
                     data-penempatan="<?php echo $job['lokasi_penempatan']; ?>"
                     data-type="<?php echo $job['job_type']; ?>"
                     data-status="<?php echo $job['is_active']; ?>"
                     data-description="<?php echo htmlspecialchars($job['description']); ?>"
                     data-requirements="<?php echo htmlspecialchars($job['requirements']); ?>"
                     data-benefits="<?php echo htmlspecialchars($job['benefits']); ?>"
                     data-information="<?php echo htmlspecialchars($job['information']); ?>"
                     data-link="<?php echo htmlspecialchars($job['link']); ?>">
                    
                    <div class="p-5">
                        <div class="flex justify-between items-start mb-3">
                            <div>
                                <h3 class="font-bold text-lg text-gray-800 job-title"><?php echo htmlspecialchars($job['title']); ?></h3>
                                <p class="text-sm text-gray-600 job-department"><?php echo htmlspecialchars($job['department']); ?></p>
                            </div>
                            <div class="<?php echo $job['is_active'] ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-500'; ?> px-2 py-1 rounded-full text-xs font-medium job-status">
                                <?php echo $job['is_active'] ? 'Aktif' : 'Tidak Aktif'; ?>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <div class="flex items-center text-sm text-gray-500 mb-1">
                                <i class="fas fa-map-marker-alt mr-2 text-gray-400"></i>
                                <span class="job-location"><?php echo htmlspecialchars($job['location']); ?></span>
                                <span class="mx-1">-</span>
                                <span class="capitalize job-penempatan"><?php echo htmlspecialchars($job['lokasi_penempatan']); ?></span>
                            </div>
                            <div class="flex items-center text-sm text-gray-500">
                                <i class="fas fa-briefcase mr-2 text-gray-400"></i>
                                <span class="capitalize job-type">
                                    <?php 
                                    $job_type_display = [
                                        'full-time' => 'Full Time',
                                        'part-time' => 'Part Time',
                                        'contract' => 'Kontrak',
                                        'internship' => 'Magang'
                                    ];
                                    echo $job_type_display[$job['job_type']] ?? $job['job_type']; 
                                    ?>
                                </span>
                            </div>
                            <?php if(!empty($job['link'])): ?>
                            <div class="flex items-center text-sm text-blue-500 mt-1">
                                <i class="fas fa-link mr-2 text-blue-400"></i>
                                <a href="<?php echo htmlspecialchars($job['link']); ?>" class="hover:underline job-link" target="_blank">Tautan Lamar</a>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <?php if(!empty($job['salary_range'])): ?>
                        <div class="flex items-center text-sm text-gray-700 font-medium mb-4">
                            <i class="fas fa-money-bill-wave mr-2 text-gray-500"></i>
                            <span class="job-salary"><?php echo htmlspecialchars($job['salary_range']); ?></span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="flex items-center text-sm text-gray-500 mb-5">
                            <i class="fas fa-users mr-2 text-gray-400"></i>
                            <span class="job-vacancy"><?php echo $job['vacancy_count']; ?> posisi tersedia</span>
                            <span class="mx-2">•</span>
                            <span class="text-xs text-gray-400 job-date">
                                Dibuat: <?php echo date('d M Y', strtotime($job['created_at'])); ?>
                            </span>
                        </div>
                        
                        <div class="flex justify-between mt-2 border-t pt-4">
                            <button class="px-3 py-1.5 bg-green-50 text-green-600 rounded hover:bg-green-100 transition-colors view-details-btn" 
                                    data-job-id="<?php echo $job['job_id']; ?>">
                                <i class="fas fa-eye mr-1"></i> Detail
                            </button>
                            <button class="px-3 py-1.5 bg-blue-50 text-blue-600 rounded hover:bg-blue-100 transition-colors edit-job-btn" 
                                    data-job-id="<?php echo $job['job_id']; ?>">
                                <i class="fas fa-edit mr-1"></i> Edit
                            </button>
                            <button class="px-3 py-1.5 bg-red-50 text-red-600 rounded hover:bg-red-100 transition-colors delete-job-btn"
                                    data-job-id="<?php echo $job['job_id']; ?>"
                                    data-job-title="<?php echo htmlspecialchars($job['title']); ?>">
                                <i class="fas fa-trash-alt mr-1"></i> Hapus
                            </button>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- No Results Message (hidden by default) -->
        <div id="no-results" class="hidden bg-white p-8 rounded-xl shadow-md text-center mt-6">
            <div class="text-gray-400 mb-4"><i class="fas fa-search text-5xl"></i></div>
            <h3 class="text-xl font-medium text-gray-700 mb-2">Tidak ada hasil yang sesuai</h3>
            <p class="text-gray-500 mb-4">Coba ubah kata kunci pencarian atau filter yang digunakan.</p>
            <button id="reset-filters" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors duration-300">
                <i class="fas fa-redo mr-2"></i> Reset Filter
            </button>
        </div>
    </div>
    
    <!-- Detail Job Modal -->
    <div id="detailJobModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-xl shadow-lg w-full max-w-3xl max-h-[90vh] overflow-y-auto">
            <div class="bg-gradient-primary p-4 text-white flex justify-between items-center sticky top-0 z-10">
                <h2 class="text-xl font-bold" id="detail-job-title">Detail Posisi Pekerjaan</h2>
                <button id="closeDetailModal" class="text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="p-6">
                <!-- Description Section (New) -->
                <div class="mb-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-3">Deskripsi Pekerjaan</h3>
                    <div id="detail-description" class="bg-gray-50 p-4 rounded-lg text-gray-700 whitespace-pre-line"></div>
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-8 mb-6">
                    <div>
                        <h3 class="text-lg font-semibold text-gray-800 mb-3">Persyaratan</h3>
                        <div id="detail-requirements" class="bg-gray-50 p-4 rounded-lg text-gray-700 whitespace-pre-line"></div>
                    </div>
                    
                    <div>
                        <h3 class="text-lg font-semibold text-gray-800 mb-3">Benefit</h3>
                        <div id="detail-benefits" class="bg-gray-50 p-4 rounded-lg text-gray-700 whitespace-pre-line"></div>
                    </div>
                </div>

                <!-- Information section (only shown if there's content) -->
                <div id="information-container" class="mb-6 hidden">
                    <h3 class="text-lg font-semibold text-gray-800 mb-3">Informasi Tambahan</h3>
                    <div id="detail-information" class="bg-gray-50 p-4 rounded-lg text-gray-700 whitespace-pre-line"></div>
                </div>
                
                <div class="mt-6 text-center">
                    <button id="closeDetailBtn" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-500">
                        <i class="fas fa-times mr-1"></i> Tutup
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Edit Job Modal -->
    <div id="editJobModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-xl shadow-lg w-full max-w-4xl max-h-[90vh] overflow-y-auto">
            <div class="bg-gradient-primary p-4 text-white flex justify-between items-center sticky top-0 z-10">
                <h2 class="text-xl font-bold">Edit Posisi Pekerjaan</h2>
                <button id="closeEditModal" class="text-white hover:text-gray-200">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="edit-job-form" class="p-6">
                <input type="hidden" id="edit_job_id" name="job_id">
                <input type="hidden" name="update_job" value="1">
                
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- Left Column -->
                    <div>
                        <div class="mb-4">
                            <label for="edit_title" class="block text-sm font-medium text-gray-700 mb-1">Judul Posisi *</label>
                            <input type="text" id="edit_title" name="title" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                        </div>

                        <div class="mb-4">
                            <label for="edit_department" class="block text-sm font-medium text-gray-700 mb-1">Departemen *</label>
                            <input type="text" id="edit_department" name="department" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                        </div>

                        <div class="mb-4">
                            <label for="edit_job_type" class="block text-sm font-medium text-gray-700 mb-1">Tipe Pekerjaan *</label>
                            <select id="edit_job_type" name="job_type" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                                <option value="full-time">Full Time</option>
                                <option value="part-time">Part Time</option>
                                <option value="contract">Kontrak</option>
                                <option value="internship">Magang</option>
                            </select>
                        </div>

                        <div class="mb-4">
                            <label for="edit_location" class="block text-sm font-medium text-gray-700 mb-1">Lokasi *</label>
                            <input type="text" id="edit_location" name="location" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                            <p class="text-xs text-gray-500 mt-1">Contoh: Jawa Timur, Jawa Tengah, dll.</p>
                        </div>

                        <div class="mb-4">
                            <label for="edit_lokasi_penempatan" class="block text-sm font-medium text-gray-700 mb-1">Lokasi Penempatan *</label>
                            <select id="edit_lokasi_penempatan" name="lokasi_penempatan" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required>
                                <option value="surabaya">Surabaya</option>
                                <option value="semarang">Semarang</option>
                            </select>
                        </div>

                        <div class="mb-4">
                            <label for="edit_salary_range" class="block text-sm font-medium text-gray-700 mb-1">Kisaran Gaji</label>
                            <input type="text" id="edit_salary_range" name="salary_range" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" placeholder="Contoh: Rp 4.500.000 - Rp 6.000.000">
                        </div>

                        <div class="mb-4">
                            <label for="edit_vacancy_count" class="block text-sm font-medium text-gray-700 mb-1">Jumlah Posisi</label>
                            <input type="number" id="edit_vacancy_count" name="vacancy_count" min="1" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500">
                        </div>

                        <div class="mb-4">
                            <label for="edit_link" class="block text-sm font-medium text-gray-700 mb-1">Link Aplikasi</label>
                            <input type="url" id="edit_link" name="link" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" placeholder="https://example.com/apply">
                            <p class="text-xs text-gray-500 mt-1">URL ke halaman aplikasi perusahaan</p>
                        </div>

                        <div class="mb-4">
                            <label class="inline-flex items-center">
                                <input type="checkbox" id="edit_is_active" name="is_active" class="form-checkbox h-5 w-5 text-green-600">
                                <span class="ml-2 text-sm text-gray-700">Aktifkan Lowongan</span>
                            </label>
                        </div>
                    </div>

                    <!-- Right Column -->
                    <div>
                        <div class="mb-4">
                            <label for="edit_description" class="block text-sm font-medium text-gray-700 mb-1">Deskripsi *</label>
                            <textarea id="edit_description" name="description" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required></textarea>
                        </div>

                        <div class="mb-4">
                            <label for="edit_requirements" class="block text-sm font-medium text-gray-700 mb-1">Persyaratan *</label>
                            <textarea id="edit_requirements" name="requirements" rows="5" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required></textarea>
                            <p class="text-xs text-gray-500 mt-1">Gunakan format list dengan awalan tanda "-" untuk setiap item</p>
                        </div>

                        <div class="mb-4">
                            <label for="edit_benefits" class="block text-sm font-medium text-gray-700 mb-1">Benefit *</label>
                            <textarea id="edit_benefits" name="benefits" rows="5" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500" required></textarea>
                            <p class="text-xs text-gray-500 mt-1">Gunakan format list dengan awalan tanda "-" untuk setiap item</p>
                        </div>

                        <div class="mb-4">
                            <label for="edit_information" class="block text-sm font-medium text-gray-700 mb-1">Informasi Tambahan</label>
                            <textarea id="edit_information" name="information" rows="4" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"></textarea>
                            <p class="text-xs text-gray-500 mt-1">Informasi tambahan tentang posisi ini</p>
                        </div>
                    </div>
                </div>

                <!-- Submit Buttons -->
                <div class="mt-6 flex justify-end">
                    <button type="button" id="cancelEdit" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-500 mr-2">
                        <i class="fas fa-times mr-1"></i> Batal
                    </button>
                    <button type="submit" id="submitEditJob" class="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-green-500">
                        <i class="fas fa-save mr-1"></i> Simpan Perubahan
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div id="deleteConfirmModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-xl shadow-lg w-full max-w-md p-6">
            <div class="text-center mb-6">
                <div class="mx-auto w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mb-4">
                    <i class="fas fa-exclamation-triangle text-red-500 text-2xl"></i>
                </div>
                <h3 class="text-xl font-semibold text-gray-800 mb-2">Konfirmasi Hapus</h3>
                <p class="text-gray-600" id="delete-confirm-text">Apakah Anda yakin ingin menghapus posisi <span id="job-title-to-delete" class="font-medium"></span>?</p>
                <p class="text-sm text-gray-500 mt-2">Tindakan ini tidak dapat dibatalkan.</p>
            </div>
            
            <form id="delete-job-form" method="post">
                <input type="hidden" id="delete_job_id" name="job_id">
                <input type="hidden" name="delete_job" value="1">
                
                <div class="flex justify-center gap-3">
                    <button type="button" id="cancelDelete" class="px-4 py-2 bg-gray-200 text-gray-700 rounded-md hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-500">
                        <i class="fas fa-times mr-1"></i> Batal
                    </button>
                    <button type="submit" id="confirmDelete" class="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500">
                        <i class="fas fa-trash-alt mr-1"></i> Hapus
                    </button>
                </div>
            </form>
        </div>
    </div>
</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Variables for search and filter
    const searchInput = document.getElementById('search-job');
    const filterPenempatan = document.getElementById('filter-penempatan');
    const filterType = document.getElementById('filter-type');
    const filterStatus = document.getElementById('filter-status');
    const resetFiltersBtn = document.getElementById('reset-filters');
    const jobCards = document.querySelectorAll('.job-card');
    const noResults = document.getElementById('no-results');
    
    // Variables for detail modal
    const detailJobModal = document.getElementById('detailJobModal');
    const detailButtons = document.querySelectorAll('.view-details-btn');
    const closeDetailModal = document.getElementById('closeDetailModal');
    const closeDetailBtn = document.getElementById('closeDetailBtn');
    const detailJobTitle = document.getElementById('detail-job-title');
    const detailDescription = document.getElementById('detail-description');
    const detailRequirements = document.getElementById('detail-requirements');
    const detailBenefits = document.getElementById('detail-benefits');
    const detailInformation = document.getElementById('detail-information');
    const informationContainer = document.getElementById('information-container');
    
    // Variables for edit modal
    const editJobModal = document.getElementById('editJobModal');
    const editButtons = document.querySelectorAll('.edit-job-btn');
    const closeEditModal = document.getElementById('closeEditModal');
    const cancelEdit = document.getElementById('cancelEdit');
    const editJobForm = document.getElementById('edit-job-form');
    
    // Variables for delete modal
   const deleteConfirmModal = document.getElementById('deleteConfirmModal');
   const deleteButtons = document.querySelectorAll('.delete-job-btn');
   const cancelDelete = document.getElementById('cancelDelete');
   const deleteJobForm = document.getElementById('delete-job-form');
   const jobTitleToDelete = document.getElementById('job-title-to-delete');
   
   // Function to show notifications
   function showNotification(message, type = 'success') {
       const notificationContainer = document.getElementById('notificationContainer');
       
       // Create notification element
       const notification = document.createElement('div');
       
       // Set appropriate classes based on type
       let bgColor, borderColor, textColor, icon;
       
       switch(type) {
           case 'success':
               bgColor = 'bg-green-100';
               borderColor = 'border-green-500';
               textColor = 'text-green-700';
               icon = 'fa-check-circle';
               break;
           case 'error':
               bgColor = 'bg-red-100';
               borderColor = 'border-red-500';
               textColor = 'text-red-700';
               icon = 'fa-exclamation-circle';
               break;
           case 'info':
               bgColor = 'bg-blue-100';
               borderColor = 'border-blue-500';
               textColor = 'text-blue-700';
               icon = 'fa-info-circle';
               break;
           default:
               bgColor = 'bg-gray-100';
               borderColor = 'border-gray-500';
               textColor = 'text-gray-700';
               icon = 'fa-bell';
       }
       
       notification.className = `${bgColor} border-l-4 ${borderColor} ${textColor} p-4 mb-4 rounded shadow-md opacity-0 transform -translate-y-4 transition-all duration-500 ease-out`;
       notification.id = `notification-${Date.now()}`;
       
       notification.innerHTML = `
           <div class="flex items-center">
               <div class="py-1">
                   <i class="fas ${icon} ${textColor} mr-2"></i>
               </div>
               <div>
                   <p class="font-bold">${type === 'success' ? 'Berhasil!' : type === 'error' ? 'Error!' : 'Info'}</p>
                   <p>${message}</p>
               </div>
               <button type="button" onclick="this.parentElement.parentElement.remove()" class="ml-auto ${textColor} hover:${textColor.replace('700', '900')}">
                   <i class="fas fa-times"></i>
               </button>
           </div>
       `;
       
       // Add to notification container
       notificationContainer.appendChild(notification);
       
       // Trigger animation to show notification
       setTimeout(() => {
           notification.classList.remove('opacity-0', 'transform', '-translate-y-4');
       }, 10);
       
       // Auto-remove after 4 seconds for success and info notifications
       if (type === 'success' || type === 'info') {
           setTimeout(() => {
               if (notification.parentElement) {
                   notification.classList.add('opacity-0', 'transform', 'translate-y-4');
                   setTimeout(() => {
                       if (notification.parentElement) {
                           notification.remove();
                       }
                   }, 500);
               }
           }, 4000);
       }
   }
   
   // Function to filter job cards
   function filterJobs() {
       const searchTerm = searchInput.value.toLowerCase();
       const selectedPenempatan = filterPenempatan.value;
       const selectedType = filterType.value;
       const selectedStatus = filterStatus.value;
       
       let visibleCount = 0;
       
       jobCards.forEach(card => {
           const title = card.dataset.title;
           const department = card.dataset.department;
           const location = card.dataset.location;
           const penempatan = card.dataset.penempatan;
           const type = card.dataset.type;
           const status = card.dataset.status;
           
           // Check if card matches all filters
           const matchesSearch = title.includes(searchTerm) || 
                                department.includes(searchTerm) || 
                                location.includes(searchTerm);
           const matchesPenempatan = selectedPenempatan === '' || penempatan === selectedPenempatan;
           const matchesType = selectedType === '' || type === selectedType;
           const matchesStatus = selectedStatus === '' || status === selectedStatus;
           
           // Show or hide card based on filter match
           if (matchesSearch && matchesPenempatan && matchesType && matchesStatus) {
               card.classList.remove('hidden');
               visibleCount++;
           } else {
               card.classList.add('hidden');
           }
       });
       
       // Show "no results" message if no cards are visible
       if (visibleCount === 0) {
           noResults.classList.remove('hidden');
       } else {
           noResults.classList.add('hidden');
       }
   }
   
   // Add event listeners for search and filters
   searchInput.addEventListener('input', filterJobs);
   filterPenempatan.addEventListener('change', filterJobs);
   filterType.addEventListener('change', filterJobs);
   filterStatus.addEventListener('change', filterJobs);
   
   // Reset filters button
   resetFiltersBtn.addEventListener('click', function() {
       searchInput.value = '';
       filterPenempatan.value = '';
       filterType.value = '';
       filterStatus.value = '';
       filterJobs();
   });
   
   // Function to show job details
   function showJobDetails(jobId) {
       const card = document.getElementById(`job-card-${jobId}`);
       if (!card) return;
       
       // Get job data from card
       const title = card.querySelector('.job-title').textContent;
       const description = card.dataset.description;
       const requirements = card.dataset.requirements;
       const benefits = card.dataset.benefits;
       const information = card.dataset.information;
       
       // Set modal content
       detailJobTitle.textContent = `Detail: ${title}`;
       detailDescription.textContent = description;
       detailRequirements.textContent = requirements;
       detailBenefits.textContent = benefits;
       
       // Handle information content (show only if there is content)
       if (information && information.trim() !== '') {
           detailInformation.textContent = information;
           informationContainer.classList.remove('hidden');
       } else {
           informationContainer.classList.add('hidden');
       }
       
       // Show modal
       detailJobModal.classList.remove('hidden');
   }
   
   // Add event listeners for detail buttons
   detailButtons.forEach(button => {
       button.addEventListener('click', function() {
           const jobId = this.dataset.jobId;
           showJobDetails(jobId);
       });
   });
   
   // Close detail modal
   closeDetailModal.addEventListener('click', function() {
       detailJobModal.classList.add('hidden');
   });
   
   closeDetailBtn.addEventListener('click', function() {
       detailJobModal.classList.add('hidden');
   });
   
   // Function to populate edit form directly from card and hidden data attributes
   function populateEditForm(jobId) {
       const card = document.getElementById(`job-card-${jobId}`);
       if (!card) return;
       
       // Basic data we can get from the card
       const title = card.querySelector('.job-title').textContent;
       const department = card.querySelector('.job-department').textContent;
       const location = card.querySelector('.job-location').textContent;
       const penempatan = card.dataset.penempatan;
       const type = card.dataset.type;
       const status = card.dataset.status === '1';
       
       // Get data from data attributes
       const description = card.dataset.description;
       const requirements = card.dataset.requirements;
       const benefits = card.dataset.benefits;
       const information = card.dataset.information;
       const link = card.dataset.link;
       
       // Set values in the form
       document.getElementById('edit_job_id').value = jobId;
       document.getElementById('edit_title').value = title;
       document.getElementById('edit_department').value = department;
       document.getElementById('edit_description').value = description;
       document.getElementById('edit_requirements').value = requirements;
       document.getElementById('edit_benefits').value = benefits;
       document.getElementById('edit_information').value = information;
       document.getElementById('edit_job_type').value = type;
       document.getElementById('edit_location').value = location;
       document.getElementById('edit_lokasi_penempatan').value = penempatan;
       document.getElementById('edit_is_active').checked = status;
       document.getElementById('edit_link').value = link;
       
       // Values we might have in the card but need to check
       const salaryEl = card.querySelector('.job-salary');
       document.getElementById('edit_salary_range').value = salaryEl ? salaryEl.textContent : '';
       
       const vacancyText = card.querySelector('.job-vacancy').textContent;
       const vacancyMatch = vacancyText.match(/(\d+)/);
       document.getElementById('edit_vacancy_count').value = vacancyMatch ? vacancyMatch[1] : '1';
       
       // Show modal
       editJobModal.classList.remove('hidden');
   }
   
   // Add event listeners for edit buttons
   editButtons.forEach(button => {
       button.addEventListener('click', function() {
           const jobId = this.dataset.jobId;
           populateEditForm(jobId);
       });
   });
   
   // Close edit modal
   closeEditModal.addEventListener('click', function() {
       editJobModal.classList.add('hidden');
   });
   
   cancelEdit.addEventListener('click', function() {
       editJobModal.classList.add('hidden');
   });
   
   // Handle edit form submission
   editJobForm.addEventListener('submit', function(e) {
       e.preventDefault();
       
       // Change button state to loading
       const submitBtn = document.getElementById('submitEditJob');
       submitBtn.disabled = true;
       submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Menyimpan...';
       
       // Collect form data
       const formData = new FormData(editJobForm);
       
       // Send AJAX request
       fetch(window.location.href, {
           method: 'POST',
           body: formData,
           headers: {
               'X-Requested-With': 'XMLHttpRequest'
           }
       })
       .then(response => response.json())
       .then(data => {
           // Reset button state
           submitBtn.disabled = false;
           submitBtn.innerHTML = '<i class="fas fa-save mr-1"></i> Simpan Perubahan';
           
           // Show notification based on response
           if (data.success) {
               showNotification(data.message, 'success');
               
               // Close modal
               editJobModal.classList.add('hidden');
               
               // Update the job card with new data
               const jobId = formData.get('job_id');
               updateJobCard(jobId, data.job);
           } else {
               showNotification(data.message, 'error');
           }
       })
       .catch(error => {
           // Reset button state and show error
           submitBtn.disabled = false;
           submitBtn.innerHTML = '<i class="fas fa-save mr-1"></i> Simpan Perubahan';
           showNotification('Terjadi kesalahan saat menyimpan data', 'error');
           console.error('Error:', error);
       });
   });
   
   // Function to update job card with new data
   function updateJobCard(jobId, job) {
       const card = document.getElementById(`job-card-${jobId}`);
       if (!card) return;
       
       // Update card content
       card.querySelector('.job-title').textContent = job.title;
       card.querySelector('.job-department').textContent = job.department;
       card.querySelector('.job-location').textContent = job.location;
       card.querySelector('.job-penempatan').textContent = job.lokasi_penempatan;
       
       // Update data attributes
       card.dataset.title = job.title.toLowerCase();
       card.dataset.department = job.department.toLowerCase();
       card.dataset.location = job.location.toLowerCase();
       card.dataset.penempatan = job.lokasi_penempatan;
       card.dataset.type = job.job_type;
       card.dataset.status = job.is_active;
       card.dataset.description = job.description;
       card.dataset.requirements = job.requirements;
       card.dataset.benefits = job.benefits;
       card.dataset.information = job.information;
       card.dataset.link = job.link;
       
       // Update job type display
       const jobTypeDisplay = {
           'full-time': 'Full Time',
           'part-time': 'Part Time',
           'contract': 'Kontrak',
           'internship': 'Magang'
       };
       card.querySelector('.job-type').textContent = jobTypeDisplay[job.job_type] || job.job_type;
       
       // Update vacancy count
       card.querySelector('.job-vacancy').textContent = `${job.vacancy_count} posisi tersedia`;
       
       // Update salary if it exists
       const salaryEl = card.querySelector('.job-salary');
       if (salaryEl && job.salary_range) {
           salaryEl.textContent = job.salary_range;
       }
       
       // Update link if it exists or create if needed
       const linkContainer = card.querySelector('.job-link');
       if (job.link) {
           if (linkContainer) {
               linkContainer.parentElement.classList.remove('hidden');
               linkContainer.textContent = 'Tautan Aplikasi';
               linkContainer.href = job.link;
           } else {
               // Create new link element if doesn't exist
               const linkDiv = document.createElement('div');
               linkDiv.className = 'flex items-center text-sm text-blue-500 mt-1';
               linkDiv.innerHTML = `
                   <i class="fas fa-link mr-2 text-blue-400"></i>
                   <a href="${job.link}" class="hover:underline job-link" target="_blank">Tautan Aplikasi</a>
               `;
               const locationDiv = card.querySelector('.job-location').parentElement.parentElement;
               locationDiv.appendChild(linkDiv);
           }
       } else if (linkContainer) {
           linkContainer.parentElement.classList.add('hidden');
       }
       
       // Update status
       const statusEl = card.querySelector('.job-status');
       if (job.is_active == 1) {
           statusEl.textContent = 'Aktif';
           statusEl.className = 'bg-green-100 text-green-700 px-2 py-1 rounded-full text-xs font-medium job-status';
       } else {
           statusEl.textContent = 'Tidak Aktif';
           statusEl.className = 'bg-gray-100 text-gray-500 px-2 py-1 rounded-full text-xs font-medium job-status';
       }
       
       // Apply a highlight effect to show the card has been updated
       card.classList.add('bg-green-50');
       setTimeout(() => {
           card.classList.remove('bg-green-50');
           card.classList.add('transition-colors', 'duration-1000');
       }, 100);
       setTimeout(() => {
           card.classList.remove('transition-colors', 'duration-1000');
       }, 1100);
   }
   
   // Add event listeners for delete buttons
   deleteButtons.forEach(button => {
       button.addEventListener('click', function() {
           const jobId = this.dataset.jobId;
           const jobTitle = this.dataset.jobTitle;
           
           // Set job ID in delete form
           document.getElementById('delete_job_id').value = jobId;
           
           // Set job title in confirmation message
           jobTitleToDelete.textContent = jobTitle;
           
           // Show delete confirmation modal
           deleteConfirmModal.classList.remove('hidden');
       });
   });
   
   // Close delete modal
   cancelDelete.addEventListener('click', function() {
       deleteConfirmModal.classList.add('hidden');
   });
   
   // Handle delete form submission
   deleteJobForm.addEventListener('submit', function(e) {
       e.preventDefault();
       
       // Change button state to loading
       const confirmBtn = document.getElementById('confirmDelete');
       confirmBtn.disabled = true;
       confirmBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-1"></i> Menghapus...';
       
       // Collect form data
       const formData = new FormData(deleteJobForm);
       
       // Send AJAX request
       fetch(window.location.href, {
           method: 'POST',
           body: formData,
           headers: {
               'X-Requested-With': 'XMLHttpRequest'
           }
       })
       .then(response => response.json())
       .then(data => {
           // Reset button state
           confirmBtn.disabled = false;
           confirmBtn.innerHTML = '<i class="fas fa-trash-alt mr-1"></i> Hapus';
           
           // Show notification based on response
           if (data.success) {
               showNotification(data.message, 'success');
               
               // Close modal
               deleteConfirmModal.classList.add('hidden');
               
               // Remove the job card with animation
               const jobId = formData.get('job_id');
               const card = document.getElementById(`job-card-${jobId}`);
               
               if (card) {
                   card.classList.add('opacity-0', 'scale-95', 'transition-all', 'duration-300');
                   setTimeout(() => {
                       card.remove();
                       
                       // Check if there are any job cards left
                       const remainingCards = document.querySelectorAll('.job-card');
                       if (remainingCards.length === 0) {
                           const grid = document.getElementById('job-positions-grid');
                           grid.innerHTML = `
                               <div class="col-span-full bg-white p-8 rounded-xl shadow-md text-center">
                                   <div class="text-gray-400 mb-4"><i class="fas fa-search text-5xl"></i></div>
                                   <h3 class="text-xl font-medium text-gray-700 mb-2">Tidak ada posisi pekerjaan</h3>
                                   <p class="text-gray-500 mb-4">Belum ada posisi pekerjaan yang dibuat. Klik tombol di bawah untuk membuat posisi baru.</p>
                                   <a href="buat_posisi_it.php" class="inline-block bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition-colors duration-300">
                                       <i class="fas fa-plus mr-2"></i> Buat Posisi Baru
                                   </a>
                               </div>
                           `;
                       }
                   }, 300);
               }
           } else {
               showNotification(data.message, 'error');
           }
       })
       .catch(error => {
           // Reset button state and show error
           confirmBtn.disabled = false;
           confirmBtn.innerHTML = '<i class="fas fa-trash-alt mr-1"></i> Hapus';
           showNotification('Terjadi kesalahan saat menghapus data', 'error');
           console.error('Error:', error);
       });
   });
   
   // Close modals when clicking outside
   window.addEventListener('click', function(e) {
       if (e.target === editJobModal) {
           editJobModal.classList.add('hidden');
       }
       if (e.target === deleteConfirmModal) {
           deleteConfirmModal.classList.add('hidden');
       }
       if (e.target === detailJobModal) {
           detailJobModal.classList.add('hidden');
       }
   });
});
</script>

<?php include '../template_it/footer.php'; ?>