<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
class BusinessCentralService
{
    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $companyId = $companies['value'][0]['id'];

        return $companyId;
    }

    public function __construct()
    {
        $this->token = $this->getAccessToken();
        $this->companyId = $this->getCompanyId($this->token);
        $this->client = new Client();
    }

    function getMetadata() {

        $customersResponse = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/\$metadata", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ]);
        $xml = simplexml_load_string((string)$customersResponse->getBody());
        $xml->registerXPathNamespace('edmx', 'http://docs.oasis-open.org/odata/ns/edmx');
        $xml->registerXPathNamespace('edm', 'http://docs.oasis-open.org/odata/ns/edm');
        $schemas = $xml->xpath('//edm:Schema');
        
        $entities = [];

        foreach ($schemas as $schema) {
            foreach ($schema->EntityType as $entity) {
                $entityName = (string) $entity['Name'];
                $properties = [];

                foreach ($entity->Property as $prop) {
                    $properties[] = [
                        'name' => (string) $prop['Name'],
                        'type' => (string) $prop['Type'],
                        'nullable' => isset($prop['Nullable']) ? (string) $prop['Nullable'] : 'true',
                    ];
                }

                $entities[$entityName] = $properties;
            }
        }

        echo "<pre>";
        echo "📦 Entity: StockkeepingUnit\n\n";

        foreach ($entities['APIStockkeeping'] as $field) {
            echo " ▸ {$field['name']} ({$field['type']})";
            echo $field['nullable'] === 'false' ? " [NOT NULL]" : "";
            echo "\n";
        }
        echo "</pre>";
        exit;
    }

    public function getAllPurchaseQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment')  and (Location_Code eq 'CI.1010') &\$select=No,Location_Code,Outstanding_Quantity";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['No'] . '_' . $line['Location_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['Outstanding_Quantity'] ?? 0);
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Purchaseqty: {$elapsed} seconds");
        

        return $result;
    }


    public function getAllTransferQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Quantity_Shipped lt Quantity and (Transfer_From_Code eq 'CI.1010'
            ) &\$select=Document_No,Item_No,Transfer_From_Code,OutstandingShipped";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['Item_No'] . '_' . $line['Transfer_from_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['OutstandingShipped'] ?? 0);
        }

        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $result;
    }

    public function getTransferLinesFromBC()
    {
        $start = microtime(true);
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Transfer_from_Code eq 'CI.1010'";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000',
            ]
        ];

        $allData = [];
        $url = $baseUrl;

        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $allData;
    }


    function getPurchaseLineFromBC() {

        $customersResponse = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment') and Location_Code eq 'CI.1010'", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ]);

        return json_decode((string)$customersResponse->getBody(), true);
    }

    function getStockkeepingFromBC()
    {
        $filter = urlencode("(Location_Code eq 'CI.1010')");
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/StockkeepingUnit?\$filter={$filter}";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        return ['value' => $allData];
    }


    public function getAllVendorsMap()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors?\$select=number,displayName";
        $vendors = [];
        do {
            $response = $this->client->get($url, [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept' => 'application/json',
                    'Prefer' => 'odata.maxpagesize=1000'
                ]
            ]);
            $data = json_decode($response->getBody(), true);

            foreach ($data['value'] as $vendor) {
                $vendors[$vendor['number']] = $vendor['displayName'];
            }

            $url = $data['@odata.nextLink'] ?? null; 

        } while ($url);

        return $vendors;
    }


    public function getVendor() {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);
        return json_decode($response->getBody(), true);
    }

    
    
    function CreatePurchaseOrder() {
        $poPayload = [
            "VendorNumber" => "VFB0011", 
            "orderDate" => date('Y-m-d')
            ];

        $response = $this->$client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($poPayload)
        ]);

        $data = json_decode((string)$response->getBody(), true);
        return $data;
    }

    function getPOFromBC() {

        $response = $this->client->get(
            "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/purchaseOrders?\$filter=number eq '30858'",
            [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept'        => 'application/json'
                ]
            ]
        );

        return json_decode((string)$response->getBody(), true);
    }

    public function getPoSuggestions($userName)
        {
            $start = microtime(true);
            $stockkeeping = $purchaseRaw = $transferRaw = $priceRaw = $vendorRaw = [];
            $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
            $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
            $transferLocationCode = "(Transfer_from_Code eq 'CI.1010' or Transfer_from_Code eq 'CI.1020')";
            $locationCode = "(Location_Code eq 'CI.1010' or Location_Code eq 'CI.1020')";
            
            $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=20000'
            ];

            $promises = [
                'stockkeeping' => $this->client->getAsync(
                    "$baseOdata/APIStockkeeping?\$filter=" . urlencode($locationCode),
                    ['headers' => $headers]
                ),

                'purchaseQtys' => $this->client->getAsync("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                        "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment') 
                        and $locationCode &\$select=No,Document_No,Approved_Date,Status,Location_Code,Outstanding_Quantity",
                        ['headers' => $headers]
                    ),

                'transferQtys' => $this->client->getAsync(
                    "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                    "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?" .
                    "\$filter=" . urlencode("Status eq 'Released' and Quantity_Shipped lt Quantity and (Transfer_from_Code eq 'CI.1010' or Transfer_from_Code eq 'CI.1020')") .
                    "&\$select=Document_No,Item_No,Transfer_From_Code,OutstandingShipped,Shipment_Date,Description",
                    ['headers' => $headers]
                ),
                'priceLists' => $this->client->getAsync(
                    "$baseOdata/Price_List_Lines?" .
                    '$filter=' . urlencode("SourceNo ne null and StartingDate le " . now()->toDateString() . " and EndingDate ge " . now()->toDateString()) .
                    '&$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo',
                    ['headers' => $headers]
                ),
                'vendors' => $this->client->getAsync(
                    "$baseApi/vendors?\$select=number,displayName",
                    ['headers' => $headers]
                ),
                'itemInvenLoc' => $this->client->getAsync(  
                    "$baseOdata/ItemInvenLoc?\$filter=(startswith(ItemNo,'FO') or startswith(ItemNo,'BV') or startswith(ItemNo,'HO') or startswith(ItemNo,'MT') or startswith(ItemNo, 'BG'))",
                    ['headers' => $headers]
                )
            ];
            
            $responses = Utils::settle($promises)->wait();
            $elapsed = microtime(true) - $start;
            Log::debug('Fetch Runtime:', ['seconds' => $elapsed]);
            if ($responses['stockkeeping']['state'] === 'fulfilled') {
                $stockkeeping = json_decode($responses['stockkeeping']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['purchaseQtys']['state'] === 'fulfilled') {
                $purchaseRaw = json_decode($responses['purchaseQtys']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['transferQtys']['state'] === 'fulfilled') {
                $transferRaw = json_decode($responses['transferQtys']['value']->getBody()->getContents(), true)['value'] ?? [];
            }

            if ($responses['priceLists']['state'] === 'fulfilled') {
                $priceRaw = json_decode($responses['priceLists']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['itemInvenLoc']['state'] === 'fulfilled') {
                $ItemInvenLoc = json_decode($responses['itemInvenLoc']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            if ($responses['vendors']['state'] === 'fulfilled') {
                $vendorRaw = json_decode($responses['vendors']['value']->getBody()->getContents(), true)['value'] ?? [];
            }
            $purchaseQtys = [];
            $purchaseDetailsMap = [];   
            foreach ($purchaseRaw as $line) {
                if (strpos($line['Document_No'], 'PRO') !== false) {
                    continue;
                }
                $key = $line['No'] . '_' . $line['Location_Code'];
                $purchaseQtys[$key] = ($purchaseQtys[$key] ?? 0) + $line['Outstanding_Quantity'];
                $purchaseDetailsMap[$key][] = [
                    'document_no' => $line['Document_No'],
                    'approved_date' => isset($line['Approved_Date']) ? date('d-m-Y', strtotime($line['Approved_Date'])) : null,
                    'status' => $line['Status'],
                    'outstanding_qty' => $line['Outstanding_Quantity']
                ];
            }
            $transferQtys = [];
            $transferDates = [];
            $transferDetailsMap = [];
            foreach ($transferRaw as $line) {
                $itemNo = $line['Item_No'];
                $location = $line['Transfer_from_Code'];
                if (!in_array($location, ['CI.1010', 'CI.1020'])) {
                    continue;
                }
                $key = $line['Item_No'] . '_' . $line['Transfer_from_Code'];
                $outstanding = $line['OutstandingShipped'] ?? 0;

                if ($outstanding > 0) {
                    $transferQtys[$key] = ($transferQtys[$key] ?? 0) + $outstanding;
                    $transferDetailsMap[$itemNo . '_' . $location][] = [
                        'document_no' => $line['Document_No'],
                        'shipment_date' => $line['Shipment_Date'],
                        'quantity' => $line['OutstandingShipped'],
                    ];
                    if (!empty($line['Shipment_Date'])) {
                        $date = $line['Shipment_Date'];
                        if (!isset($transferDates[$key]) || $date < $transferDates[$key]) {
                            $transferDates[$key] = $date;
                        }
                    }
                }
            }


            $priceMap = [];
            foreach ($priceRaw as $line) {
                if (!empty($line['Asset_No']) && !empty($line['SourceNo'])) {
                    $priceMap[$line['Asset_No']][$line['SourceNo']] = true;
                }
            }
            foreach ($priceMap as $itemNo => $vendorList) {
                $priceMap[$itemNo] = array_keys($vendorList);
            }

            $unitCostsByItem = []; 
            foreach ($priceRaw as $line) {
                if (!empty($line['Asset_No']) && !empty($line['SourceNo']) && isset($line['DirectUnitCost'])) {
                    $itemNo = $line['Asset_No'];
                    $vendorNo = $line['SourceNo'];
                    $unitCostsByItem[$itemNo][$vendorNo] = $line['DirectUnitCost'];
                }
            }

            $vendorMap = [];
            foreach ($vendorRaw as $v) {
                $vendorMap[$v['number']] = $v['displayName'];
            }
            $skuKeys = [];
            foreach ($stockkeeping as $sku) {
                $skuKeys[$sku['Item_No'] . '_' . $sku['Location_Code']] = true;
            }
            $itemInvenKey = [];
            foreach ($ItemInvenLoc as $i => $item) {
                $itemInvenKey[$item['ItemNo'] . '_' . $item['Location']] = $i;
            }
            $results = [];
            foreach ($stockkeeping as $sku) {
                $itemNo = $sku['Item_No'];
                $location = $sku['Location_Code'];
                $stock = $sku['Inventory'];
                $minQty = $sku['MinInven'];
                $maxQty = $sku['MaxInven'];
                $description = $sku['Description'];
                $active = $sku['Active'];
                $comment = $sku['Comment'];
                $dbStatus = $sku['Status'];
                $key = $itemNo . '_' . $location;
                $normalizedDescription = strtoupper(preg_replace('/[^A-Z0-9]/', '', $description));

                if ($location === 'CI.1010' && str_contains($normalizedDescription, 'FRUITVEGETABLE')) {
                    continue;
                }
                $onPO = $purchaseQtys[$key] ?? 0;
                $inTransfer = $transferQtys[$key] ?? 0;
                $qtyToOrder = max(0, ceil(($maxQty + $inTransfer) - ($stock + $onPO)));
                $vendorNos = $priceMap[$itemNo] ?? [];
                $transferDate = $transferDates[$key] ?? null;
                
                if ($qtyToOrder <= 0) {
                    $status = "No Need Order";
                }
                if ($qtyToOrder > 0) {
                    if ( ($stock + $onPO) - ($inTransfer) < $minQty ) {
                        $status = "Need Order";
                        if ($inTransfer > 0) {
                            $status = "Need Order For TO";
                        }
                    } else
                    {
                        $status = "Order For Stock";
                    }
                }
                if ($onPO > 0 && $stock < ($minQty+$inTransfer) && ($stock+$onPO) > ($minQty+$inTransfer)){
                    $status = "Follow Up PO";
                }

                if (empty($vendorNos)) {
                $results[] = [  
                    'item_no'       => $itemNo,
                    'description'   => $description,
                    'stock'         => $stock,
                    'min_qty'       => $minQty,
                    'max_qty'       => $maxQty,
                    'on_po'         => $onPO,
                    'in_transfer'   => $inTransfer,
                    'qty_to_order'  => $qtyToOrder,
                    'vendor_name'   => 'No Vendor',
                    'vendor_no'     => '0',
                    'location_code' => $location,
                    'unit_cost' => '0',
                    'status'        => $status,
                    'transfer_lines'=> $transferDetailsMap[$key] ?? [],
                    'po_lines' => $purchaseDetailsMap[$key] ?? [],
                    'Active' => $active,
                    'Comment' => $comment,
                    'Status' => $dbStatus
                ];
            } else {
                foreach ($vendorNos as $vendorNo) {
                    $results[] = [
                        'item_no'       => $itemNo,
                        'description'   => $description,
                        'stock'         => $stock,
                        'min_qty'       => $minQty,
                        'max_qty'       => $maxQty,
                        'on_po'         => $onPO,
                        'in_transfer'   => $inTransfer,
                        'qty_to_order'  => $qtyToOrder,
                        'vendor_name'   => $vendorMap[$vendorNo] ?? null,
                        'vendor_no'     => $vendorNo,
                        'location_code' => $location,
                        'unit_cost' => $unitCostsByItem[$itemNo][$vendorNo],
                        'status'        => $status,
                        'transfer_lines'=> $transferDetailsMap[$key] ?? [],
                        'po_lines' => $purchaseDetailsMap[$key] ?? [],
                        'Active' => $active,
                        'Comment' => $comment,
                        'Status' => $dbStatus
                    ];
                }
                }

            }

            foreach ($transferRaw as $line) {
                $itemNo = $line['Item_No'];
                $location = $line['Transfer_from_Code'];
                $key = $itemNo . '_' . $location;
                if (!in_array($location, ['CI.1010', 'CI.1020'])) {
                    continue;
                }
                if (isset($skuKeys[$key])) {
                    continue;
                }
                $onPO = $purchaseQtys[$key] ?? 0;
                $inTransfer = $transferQtys[$key] ?? 0;
                $transferDate = $line['Shipment_Date'] ?? null;

                $description = $line['Description'] ?? '';
                $normalizedDescription = strtoupper(preg_replace('/[^A-Z0-9]/', '', $description));
                if ($location === 'CI.1010' && str_contains($normalizedDescription, 'FRUITVEGETABLE')) {
                    continue;
                }
                
                $stock = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Inventory'] : 0;
                $Active = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Active'] : true;
                $Comment = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Comment'] : '';
                $dbStatus = isset($itemInvenKey[$key]) ? $ItemInvenLoc[$itemInvenKey[$key]]['Status'] : '__';
                $minQty = 0;
                $maxQty = 0;
                $qtyToOrder = max(0, ceil($maxQty + $inTransfer - ($stock + $onPO)));

                if ($qtyToOrder <= 0) {
                    $status = "No Need Order";
                } else {
                    if (($stock + $onPO) - $inTransfer < $minQty) {
                        $status = "Need Order";
                        if ($inTransfer > 0) {
                                $status = "Need Order For TO";
                        }
                    } else {
                        $status = "Order For Stock";
                    }
                }
                if ($onPO > 0 && $stock < ($minQty+$inTransfer) && ($stock+$onPO) > ($minQty+$inTransfer)){
                    $status = "Follow Up PO";
                }

                $vendorNos = $priceMap[$itemNo] ?? [];

                if (empty($vendorNos)) {
                    $results[] = [
                        'item_no'       => $itemNo,
                        'description'   => $description,
                        'stock'         => $stock,
                        'min_qty'       => $minQty,
                        'max_qty'       => $maxQty,
                        'on_po'         => $onPO,
                        'in_transfer'   => $inTransfer,
                        'qty_to_order'  => $qtyToOrder,
                        'vendor_name'   => 'No Vendor',
                        'vendor_no'     => '0',
                        'location_code' => $location,
                        'unit_cost'     => '0',
                        'status'        => $status,
                        'transfer_lines' => $transferDetailsMap[$key] ?? [],
                        'po_lines' => $purchaseDetailsMap[$key] ?? [],
                        'Active' => $active,
                        'Comment' => $Comment,
                        'Status' => $dbStatus
                        
                    ];
                } else {
                    foreach ($vendorNos as $vendorNo) {
                        $results[] = [
                            'item_no'       => $itemNo,
                            'description'   => $description,
                            'stock'         => $stock,
                            'min_qty'       => $minQty,
                            'max_qty'       => $maxQty,
                            'on_po'         => $onPO,
                            'in_transfer'   => $inTransfer,
                            'qty_to_order'  => $qtyToOrder,
                            'vendor_name'   => $vendorMap[$vendorNo] ?? null,
                            'vendor_no'     => $vendorNo,
                            'location_code' => $location,
                            'unit_cost'     => $unitCostsByItem[$itemNo][$vendorNo] ?? 0,
                            'status'        => $status,
                            'transfer_lines' => $transferDetailsMap[$key] ?? [],
                            'po_lines' => $purchaseDetailsMap[$key] ?? [],
                            'Active' => $active,
                            'Comment' => $Comment,
                            'Status' => $dbStatus
                        ];
                    }
                }
            }

            $results = array_filter($results, function ($item) {
                $desc = strtoupper(preg_replace('/[^A-Z0-9]/', '', $item['description'] ?? ''));
                return !(($item['location_code'] === 'CI.1010') && str_contains($desc, 'FRUITVEGETABLE'));
            });
            return [
                'items' => $results,
                'vendors' => $vendorMap
            ];
        }


    public function getItemIdByNo(string $itemNo)
    {
        $filter = urlencode("number eq '$itemNo'");
        $response = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
       "/api/v2.0/companies({$this->companyId})/items?\$filter=$filter", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ]
        ]);

        $items = json_decode($response->getBody(), true);
        return $items['value'][0]['id'] ?? null;
    }


    public function createPurchaseOrderForVendor($vendorNo, $userName)
    {
        $purchaser = in_array($userName, [
                'titania@citbi.onmicrosoft.com',
                'adminbc@citbi.onmicrosoft.com'
            ]) ? 'TN01' : 'UT01';

        $payload = [
            'vendorNumber' => $vendorNo,
            'orderDate' => now()->toDateString(),
            'purchaser' => $purchaser,
            'shortcutDimension2Code' => "100000",
            'requestedReceiptDate' => today()->toDateString()
        ];

        $response = $this->client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payload)
        ]);
        $body = json_decode($response->getBody(), true);
        return json_decode($response->getBody(), true);
    }

    public function getLocationIdByCode($locationCode)
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/locations?\$filter=code eq '{$locationCode}'";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);

        $data = json_decode((string)$response->getBody(), true);
        
        return $data['value'][0]['id'] ?? null;
    }


    public function addPurchaseLineToPO($poId, $itemNo, $quantity, $cost, $userName)
    {   $start = microtime(true);
        $itemId = $this->getItemIdByNo($itemNo);
        $locationCode = in_array($userName, [
                'titania@citbi.onmicrosoft.com',
                'adminbc@citbi.onmicrosoft.com'
            ]) ? 'CI.1010' : 'CI.1020';
        if (str_contains($itemNo, 'BV') && $cost > 1000000) {
            $locationCode = 'CI.1051';
        }
        $locationId = $this->getLocationIdByCode($locationCode);
        $payload = json_encode([
            'itemId' => $itemId,
            'quantity' => (float) $quantity,
            'locationId' => $locationId
        ]);

        $response = $this->client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders({$poId})/purchaseOrderLines", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ],
            'body' => $payload
        ]);
        $elapsed = microtime(true) - $start;
        Log::info("Create POLine runtime: {$elapsed} seconds");
        return json_decode($response->getBody(), true);
    }

    public function getAsyncAddPurchaseLinePromise($poId, $itemNo, $quantity, $userName)
    {   
        $start = microtime(true);
        $itemId     = $this->getItemIdByNo($itemNo);
        $locationId = $this->getLocationIdByCode(in_array($userName, [
                'titania@citbi.onmicrosoft.com',
                'adminbc@citbi.onmicrosoft.com'
            ]) ? 'CI.1010' : 'CI.1020');

        $payload = [
            'itemId'     => $itemId,
            'quantity'   => (float) $quantity,
            'locationId' => $locationId,
        ];
        $elapsed = microtime(true) - $start;
        Log::info("Create POLine runtime: {$elapsed} seconds");
        return $this->client->postAsync(
            "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})/purchaseOrders({$poId})/purchaseOrderLines",
            [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept'        => 'application/json',
                    'Content-Type'  => 'application/json',
                ],
                'body' => json_encode($payload),
            ]
        );
    }



    public function getAllPurchasePrices()
    {
        $start = microtime(true);
        $today = now()->toDateString(); // e.g. "2025-06-17"
        $filter = "SourceNo ne null and StartingDate le $today and EndingDate ge $today";
        $encodedFilter = urlencode($filter);

        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];

        try {
            do {
                $response = $this->client->get($url, $headers);
                $decoded = json_decode((string)$response->getBody(), true);

                if (isset($decoded['value'])) {
                    $allData = array_merge($allData, $decoded['value']);
                }

                $url = $decoded['@odata.nextLink'] ?? null;
            } while ($url);

            return collect($allData)->groupBy('Asset_No');
        } catch (\Exception $e) {
            Log::error("Failed to fetch Price List Lines: " . $e->getMessage());
            return collect(); 
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Priccss: {$elapsed} seconds");
        
    }
}
