<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\BusinessCentralService;
use App\Services\InventoryBC;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use GuzzleHttp\Client;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;

class BusinessCentralController extends Controller
{   

    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $companyId = $companies['value'][0]['id'];

        return $companyId;
    }

    public function __construct()
    {
        $this->token = $this->getAccessToken();
        $this->companyId = $this->getCompanyId($this->token);
        $this->client = new Client();
    }


    public function createAndPostPO()
    {
        $bc = new BusinessCentralService();
        $purchaseOrder= $bc->CreatePurchaseOrder();
        
        return response()->json($purchaseOrder);
    }

    public function showPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $service = new BusinessCentralService();

        $cacheKey = 'po_suggestions_' . md5($userName);
        if (Cache::has($cacheKey)) {
            Log::info("CACHE HIT: $cacheKey");
        } else {
            Log::info("CACHE MISS: $cacheKey");
        }
        $poData = Cache::remember($cacheKey, now()->addMinutes(120), function () use ($userName) {
            $service = new BusinessCentralService();
            return $service->getPoSuggestions($userName);
        });

        $allItems = $poData['items'];
        $vendorMap = $poData['vendors'];
        $vendorOptionsByItemNo = [];
        $uniqueItems = [];
        foreach ($allItems as $item) {
            $itemNo = $item['item_no'] ?? null;
            $vendorNo = $item['vendor_no'] ?? null;
            $vendorName = $item['vendor_name'] ?? null;
            $unitCost = $item['unit_cost'] ?? null;
            if ($vendorNo && $vendorName) {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => $vendorName,
                    'unit_cost' => $unitCost,
                ];
            } elseif ($vendorNo === '0') {
                $vendorOptionsByItemNo[$itemNo][$vendorNo] = [
                    'name' => 'No Vendor',
                    'unit_cost' => 0,
                ];
            }

            if (!isset($uniqueItems[$itemNo])) {
                $uniqueItems[$itemNo] = $item;
            }
        }

        $vendorSet = collect($allItems)
            ->filter(fn ($item) => $item['vendor_no'] && $item['vendor_name'])
            ->pluck('vendor_name', 'vendor_no')
            ->unique()
            ->sortBy(fn ($name, $no) => $name)
            ->toArray();

        $locationSet = collect($allItems)->pluck('location_code')->filter()->unique()->sort()->values()->all();

        $statusSet = collect($allItems)
        ->pluck('status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        $statusItemSet = collect($allItems)
        ->pluck('Status')
        ->filter()
        ->unique()
        ->sort()
        ->values()
        ->all();

        
        session([
            'po_items' => array_values($uniqueItems),
            'vendor_map' => $vendorMap
        ]);

        return view('po-suggestion', [
            'items'                => array_values($uniqueItems),
            'vendors'              => $vendorSet,
            'locations'            => $locationSet,
            'vendorOptionsByItem'  => $vendorOptionsByItemNo,
            'statuses'             => $statusSet,
            'statusItems'           => $statusItemSet
        ]);
    }

    public function refreshPoSuggestions(Request $request)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;

        $cacheKey = 'po_suggestions_' . md5($userName);
        Cache::forget($cacheKey);

        return redirect()->route('po-suggestions')->with('success', 'Cache has been refreshed.');
    }

    public function refreshMinMaxSuggestions(Request $request)
    {
        Cache::forget('sku_mapping_cache');
       return redirect()->route('sku-mapping')->with('success', 'Cache has been refreshed.');
    }


    public function viewSkuMapping()
    {
        $start = microtime(true);

        $bcService = new InventoryBC();
        $data = $bcService->getItemLedgerAndSkuMappingAsync();

        $executionTime = round(microtime(true) - $start, 2);
        \Log::info("viewSkuMapping executed in {$executionTime} seconds");
        return view('sku-mapping', [
            'result' => $data['result'],
            'executionTime' => $executionTime,
        ]);
    }


    public function exportMissingVendor(Request $request)
    {
        $start = microtime(true);
        $user = session('user');
        $userName = $user['email'] ?? null;
        $items = session('po_items', []);
        $locationCode = $request->query('location_code');
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
        $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ];
        $step1 = microtime(true);
       

        $today = now()->toDateString();
        $items = collect($items); 
        if ($locationCode) {
            $items = $items->filter(function ($item) use ($locationCode) {
                return isset($item['location_code']) && $item['location_code'] === $locationCode;
            });
        }
         $missingVendorItems = collect($items)->filter(function ($item) {
            return empty($item['vendor_no']) || $item['vendor_no'] == 0;
        });
        $itemNos = $items->pluck('item_no')->unique()->filter()->values();
        $priceList = [];

        $step2 = microtime(true);   
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000'
            ]
        ];

        foreach ($itemNos->chunk(200) as $chunk) {
            $assetFilter = $chunk->map(fn($no) => "Asset_No eq '$no'")->implode(' or ');
            $filter = "SourceNo ne '' and StartingDate le $today and EndingDate lt $today and ($assetFilter)";
            $encodedFilter = urlencode($filter);

            $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
                "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

            try {
                $response = $this->client->get($url, $headers);
                $data = json_decode((string)$response->getBody(), true);
                $priceList = array_merge($priceList, $data['value'] ?? []);
            } catch (\Throwable $e) {
                \Log::error("Failed to fetch price list chunk: " . $e->getMessage());
            }
        }

        
        
        $vendorNameMap = session('vendor_map', []);
        
        $inactiveVendorsByItem = collect($priceList)->groupBy('Asset_No');
        
         return Excel::download(new class($missingVendorItems, $inactiveVendorsByItem, $vendorNameMap) implements FromCollection, WithHeadings {
            private $items, $vendorMap, $vendorNameMap;

            public function __construct($items, $vendorMap, $vendorNameMap)
            {
                $this->items = $items;
                $this->vendorMap = $vendorMap;
                $this->vendorNameMap = $vendorNameMap;
            }

            public function collection()
            {
                return $this->items->map(function ($item) {
                    $itemNo = $item['item_no'] ?? '';
                    $vendorInfo = $this->vendorMap[$itemNo][0] ?? null;
                    $vendorNo = $vendorInfo['AssignToNo'] ?? '';
                    $vendorName = $this->vendorNameMap[$vendorNo] ?? 'N/A';

                    return [
                        'Kode'     => $itemNo,
                        'Item' => $item['description'] ?? '',
                        'Supplier' => $vendorName,
                        'Harga Non PPN'   => $vendorInfo['DirectUnitCost'] ?? 'N/A',
                        'UoM'   => $vendorInfo['Unit_of_Measure_Code'] ?? 'N/A'
                    ];
                });
            }

            public function headings(): array
            {
                return ['Kode', 'Item', 'Supplier', 'Harga Non PPN', 'UoM'];
            }
        }, 'missing-vendors.xlsx');
    }
    
   public function exportSKUByLocation(Request $request)
    {   
        $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
        $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";
        $headers = [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=20000'
            ];
        $twoMonthsAgo = now()->subMonths(2)->toDateString();
        $promise = $this->client->getAsync(
            "$baseOdata/Price_List_Lines?" .
            '$filter=' . urlencode("SourceNo ne null and StartingDate le " . now()->toDateString() . " and EndingDate ge " . $twoMonthsAgo) .
            '&$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo',
            ['headers' => $headers]
        );
        
        $uomPromise = $this->client->getAsync(
            "$baseOdata/Item_UOM?" .
            '$select=Item_No,Code,Qty_per_Unit_of_Measure',
            ['headers' => $headers]
        );
        
        $priceListResponse = $promise->wait();
        $uomResponse = $uomPromise->wait();
    
        $priceLists = collect(json_decode($priceListResponse->getBody()->getContents(), true)['value'] ?? []);
        $uomList = collect(json_decode($uomResponse->getBody()->getContents(), true)['value'] ?? []);

        $priceMap = $priceLists
        ->groupBy('Asset_No')
        ->map(function ($group) {
            return collect($group)->sortByDesc('DirectUnitCost')->first();
        });
        $uomMap = $uomList->keyBy(function ($item) {
            return $item['Item_No'] . '|' . $item['Code'];
        });
        $rawRows = $request->input('rows', []);
        if (is_string($rawRows)) {
            $rows = json_decode($rawRows, true);
        } elseif (is_array($rawRows) && is_string($rawRows[0])) {
            $rows = json_decode($rawRows[0], true);
        } else {
            $rows = $rawRows;
        }
        $rows = collect($rows);
        $finalData = $rows->map(function ($row) use ($priceMap, $uomMap) {
            if (is_object($row)) $row = (array) $row;
    
            $itemNo = $row['Item_No'] ?? '';
            $price = $priceMap->get($itemNo);
            $unitCode = data_get($price, 'Unit_of_Measure_Code', '');
    
            $unitCost = is_numeric(data_get($price, 'DirectUnitCost')) ? (float) data_get($price, 'DirectUnitCost') : 0;
            $uomData = $uomMap->get($itemNo . '|' . $unitCode);
            $qtyPerUom = isset($uomData['Qty_per_Unit_of_Measure']) ? (float) $uomData['Qty_per_Unit_of_Measure'] : 1;
            $pricePerBaseUnit = $unitCost / $qtyPerUom;
            $realMaxQty = (float) ($row['RealMaxQty'] ?? 0);
            $stockFor = (float) ($row['Stock_For'] ?? 0);
    
            $total = $realMaxQty * $pricePerBaseUnit * (30 / $stockFor);
    
            return [
                'Item_No'        => $row['Item_No'] ?? '',
                'Description'    => $row['Description'] ?? '',
                'RealMinQty'     => $row['RealMinQty'] ?? 0,
                'RealMaxQty'     => $realMaxQty,
                'Stock_For'      => $stockFor,
                'HargaTertinggi' => $unitCost,
                'UoM'            => $unitCode,
                'UoM_Factor'     => $qtyPerUom,
                'Total'          => $total,
                'Location_Code'  => $row['Location'] ?? 'Unknown',
            ];
        });
    
        $grouped = $finalData->groupBy('Location_Code');
    
        return Excel::download(new class($grouped) implements WithMultipleSheets {
            private $groupedData;
    
            public function __construct(Collection $groupedData)
            {
                $this->groupedData = $groupedData;
            }
    
            public function sheets(): array
            {
                $sheets = [];
    
                foreach ($this->groupedData as $location => $dataRows) {
                    $sheets[] = new class($location, $dataRows) implements FromCollection, WithHeadings {
                        private $location;
                        private $data;
    
                        public function __construct($location, $data)
                        {
                            $this->location = $location;
                            $this->data = $data;
                        }
    
                        public function collection()
                        {
                            return collect($this->data);
                        }
    
                        public function headings(): array
                        {
                            return array_keys($this->data[0] ?? []);
                        }
    
                        public function title(): string
                        {
                            return substr($this->location, 0, 31);
                        }
                    };
                }
    
                return $sheets;
            }
        }, 'SKU_Export_By_Location_' . now()->format('Ymd_His') . '.xlsx');

    }

    public function patchItemField(Request $request)
    {
        $itemNo = $request->input('item_no');
        $variantCode = $request->input('variant_code', '');
        $location = $request->input('location_code');

        $fieldsToUpdate = [];
        $fieldaToUpdate = [];

        if ($request->has('Status')) {
            $fieldsToUpdate['Status'] = $request->input('Status');
            $fieldaToUpdate['Status'] = $request->input('Status');
        }
        if ($request->has('Comment')) {
            $fieldsToUpdate['Comment'] = $request->input('Comment');
            $fieldaToUpdate['Comment'] = $request->input('Comment');
        }
        if ($request->has('Active')) {
            $fieldsToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
            $fieldaToUpdate['Active'] = filter_var($request->input('Active'), FILTER_VALIDATE_BOOLEAN);
        }
        
        

        try {
            $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
            $company = "/ODataV4/Company('" . env('BC_COMPANY_ID') . "')/APIStockkeeping";
            $patchUrl = $baseUrl . $company .
                "(Item_No='" . rawurlencode($itemNo) . "'," .
                "Variant_Code='" . rawurlencode($variantCode) . "'," .
                "Location_Code='" . rawurlencode($location) . "')";

            $headers = [
                'Authorization' => "Bearer " . $this->getAccessToken(),
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
                'If-Match' => '*'
            ];

            $response = $this->client->request('PATCH', $patchUrl, [
                'headers' => $headers,
                'body' => json_encode($fieldsToUpdate)
            ]);

            return response()->json(['message' => 'Item updated.']);
        } catch (\Exception $e) {
             \Log::warning("PATCH to APIStockkeeping failed for $itemNo at $location: " . $e->getMessage());
            
            try {
                $fallbackUrl = $baseUrl . "/ODataV4/Company('$this->companyId')/ItemInvenLoc" .
                    "(ItemNo='" . rawurlencode($itemNo) . "'," .
                    "Location='" . rawurlencode($location) . "')";

                $this->client->request('PATCH', $fallbackUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($fieldaToUpdate)
                ]);
                \Log::info("Fallback PATCH succeeded for $itemNo at $location");
                return response()->json("message' => 'Item updated.");
            } catch (\Exception $fallbackError) {
                \Log::error("Both APIStockkeeping and fallback ItemInvenLoc PATCH failed for $itemNo at $location: " . $fallbackError->getMessage());
                return response()->json(['error' => 'Failed to update item.'], 500);
            }
            return response()->json(['error' => 'Failed to update item.'], 500);
        }
    }



    public function create(Request $request, BusinessCentralService $bcService)
    {
        $user = session('user');
        $userName = $user['email'] ?? null;
        $selectedItems = array_unique($request->input('selected_items', []));
        $vendorMap = $request->input('vendor_no', []);
        $qtyMap = $request->input('qty_to_order', []);
        $unitCost = $request->input('unit_cost', []);

        $groupedByVendor = [];

        foreach ($selectedItems as $itemNo) {
            $vendor = $vendorMap[$itemNo] ?? null;
            $qty = $qtyMap[$itemNo] ?? 0;
            $cost = $unitCost[$itemNo];

            if (!$vendor || $vendor === '0' || $qty <= 0) continue;

            $groupedByVendor[$vendor][] = [
                'item_no' => $itemNo,
                'quantity' => $qty,
                'cost' => $cost,
            ];
        }
        $results = [];
        
        foreach ($groupedByVendor as $vendorNo => $items) {
            $po = $bcService->createPurchaseOrderForVendor($vendorNo, $userName);

            $promises = [];

            foreach ($items as $line) {
                $response = $bcService->addPurchaseLineToPO(
                    $po['id'],
                    $line['item_no'],
                    $line['quantity'],
                    $line['cost'],
                    $userName 
                );
            }

            $results[] = [
                'vendor'     => $vendorNo,
                'po_no'      => $po['number'] ?? 'N/A',
                'line_count' => count($items),
            ];
        }
    
        return redirect()->back()->with('success', 'Purchase Orders created successfully.')->with('createdPOs', $results);
    }

    public function updateSku(Request $request)
    {
        set_time_limit(3000);
        $updatedRows = $request->input('rows', []);

        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/SKUImport";

        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($updatedRows as $row) {
            $patchUrl = $baseUrl . $company . "(Item_No='" . rawurlencode($row['Item_No']) . "',Location='" . rawurlencode($row['Location']) . "')";

            $body = [
                'StockFor' => $row['Stock_For'],
                'LeadTime' => $row['Lead Time'],
                'MinInven' => $row['ActMinQty'],
                'MaxInven' => $row['ActMaxQty'],
                'Comment' => $row['Comment'] ?? ''
            ];

            try {
                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body' => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::warning("PATCH failed for SKU {$row['Item_No']}, trying POST. Reason: " . $e->getMessage());

                try {
                    $this->client->request('POST', $baseUrl . $company, [
                        'headers' => $headers,
                        'body' => json_encode(array_merge([
                            'Item_No' => $row['Item_No'],
                            'Location' => $row['Location']
                        ], $body))
                    ]);
                } catch (\Exception $e2) {
                    \Log::error("Failed to POST SKU {$row['Item_No']}: " . $e2->getMessage());
                }
            }
        }

        return response()->json(['message' => 'SKU update process completed']);
    }


    public function updateStockkeeping(Request $request)
    {
        set_time_limit(3000);

        $selectedRows = $request->input('rows', []);

        $this->updateSku($request);

        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')/APIStockkeeping";

        $headers = [
            'Authorization' => "Bearer {$this->token}", 
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
            'If-Match' => '*'
        ];

        foreach ($selectedRows as $row) {
            try {
                $patchUrl = $baseUrl . $company .
                    "(Item_No='" . rawurlencode($row['Item_No']) . "'," .
                    "Variant_Code='" . rawurlencode($row['Variant_Code'] ?? '') . "'," .
                    "Location_Code='" . rawurlencode($row['Location']) . "')";

                $body = [
                    'MinInven' => $row['ActMinQty'],
                    'MaxInven' => $row['ActMaxQty']
                ];

                $this->client->request('PATCH', $patchUrl, [
                    'headers' => $headers,
                    'body'    => json_encode($body)
                ]);
            } catch (\Exception $e) {
                \Log::error("Failed to patch Stockkeeping for {$row['Item_No']} - {$row['Location']}: " . $e->getMessage());
                return response()->json(['error' => 'Stockkeeping update failed. See logs.'], 500);
            }
        }

        return response()->json(['message' => 'Stockkeeping updated successfully']);
    }




}
