<?php
// Start session
session_start();

// Include database connection and helper functions
require_once "../config/connection.php";
require_once "transaction_actions_it.php";

// Redirect if not IT user
if (!is_logged_in() || !is_user_type('it')) {
    header("Location: ../login.php");
    exit();
}

// Get IT user ID
$it_id = $_SESSION['user_id'];

// UPDATED: Update overdue status for borrowed items automatically
updateOverdueStatus($conn);

// Log activity
log_activity($conn, $it_id, 'it', 'Approval Form Access', 'IT user accessed borrowing approval page');

// Set page title
$page_title = "Persetujuan Formulir Peminjaman - IT";

// FIXED: Handle AJAX actions with proper data validation
if (isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    // Debug logging for bulk actions
    if (isset($_POST['bulk_action']) && $_POST['bulk_action'] == 'true') {
        error_log("=== BULK ACTION DEBUG ===");
        error_log("Action: " . $_POST['action']);
        error_log("Item IDs: " . $_POST['item_ids']);
        error_log("Transaction ID: " . $_POST['transaction_id']);
        error_log("Bulk Action Flag: " . $_POST['bulk_action']);
        error_log("POST Data: " . json_encode($_POST));
    }
    
    // FIXED: Handle bulk actions with validation
    if (isset($_POST['bulk_action']) && $_POST['bulk_action'] == 'true') {
        // Validate required fields for bulk action
        if (empty($_POST['item_ids']) || empty($_POST['transaction_id']) || empty($_POST['action'])) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Data tidak lengkap untuk bulk action'
            ]);
            exit();
        }
        
        $response = handleBulkTransactionAction($conn);
    } else {
        // Validate required fields for single action
        if (empty($_POST['item_id']) || empty($_POST['transaction_id']) || empty($_POST['action'])) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Data tidak lengkap untuk single action'
            ]);
            exit();
        }
        
        $response = handleTransactionAction($conn);
    }
    
    echo json_encode($response);
    exit();
}

// UPDATED: Check for show_reservation parameter
$show_reservation = isset($_GET['show_reservation']) ? $_GET['show_reservation'] : '';

// Prepare filters
$filter_borrower = isset($_GET['borrower']) ? $_GET['borrower'] : '';
$filter_status = isset($_GET['status']) ? $_GET['status'] : '';
$filter_date_start = isset($_GET['date_start']) ? $_GET['date_start'] : '';
$filter_date_end = isset($_GET['date_end']) ? $_GET['date_end'] : '';
$tab = isset($_GET['tab']) ? $_GET['tab'] : 'process'; // Default to 'process' tab

// Get transactions for current tab
$transactions = getTransactions($conn, $tab, $filter_borrower, $filter_status, $filter_date_start, $filter_date_end);

// UPDATED: If show_reservation parameter exists, find the transaction ID
$auto_show_transaction_id = null;
if (!empty($show_reservation)) {
    $reservation_stmt = $conn->prepare("SELECT id FROM stock_transactions WHERE reservation_number = ?");
    $reservation_stmt->bind_param("s", $show_reservation);
    $reservation_stmt->execute();
    $reservation_result = $reservation_stmt->get_result();
    if ($reservation_result->num_rows > 0) {
        $auto_show_transaction_id = $reservation_result->fetch_assoc()['id'];
    }
    $reservation_stmt->close();
}

// Get borrowers for filter dropdown
$borrowers_query = "SELECT * FROM borrower_users ORDER BY username";
$borrowers_result = $conn->query($borrowers_query);

// Include IT header
include '../template_it/header.php';
?>

<!-- Main Content Area -->
<main class="flex-1 p-6 overflow-auto bg-gradient-to-br from-purple-100 via-blue-100 to-indigo-100">
    <!-- CSS for transitions -->
    <style>
        .transition-all {
            transition-property: all;
            transition-timing-function: cubic-bezier(0.4, 0, 0.2, 1);
        }

        .duration-1000 {
            transition-duration: 1000ms;
        }

        .updated-item {
            animation: highlight-fade 2s ease-out;
        }

        @keyframes highlight-fade {
            0% { background-color: rgba(254, 249, 195, 1); } /* Yellow */
            100% { background-color: rgba(249, 250, 251, 1); } /* Gray-50 */
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        .tab-button {
            transition: all 0.3s ease;
        }

        .tab-button.active {
            background-color: rgb(147 51 234);
            color: white;
        }

        .tab-button:not(.active) {
            background-color: rgb(229 231 235);
            color: rgb(75 85 99);
        }

        .tab-button:not(.active):hover {
            background-color: rgb(209 213 219);
        }

        /* UPDATED: Add highlight styles for target reservation */
        .highlight-reservation {
            background-color: rgba(254, 249, 195, 1) !important;
            border-left: 4px solid #f59e0b !important;
            animation: pulse-highlight 3s ease-out;
        }

        @keyframes pulse-highlight {
            0% { 
                background-color: rgba(254, 249, 195, 1);
                transform: scale(1);
            }
            50% { 
                background-color: rgba(251, 191, 36, 0.3);
                transform: scale(1.02);
            }
            100% { 
                background-color: rgba(254, 249, 195, 1);
                transform: scale(1);
            }
        }

        /* NEW: Bulk action styles */
        .bulk-action-bar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            display: none;
        }

        .bulk-action-bar.show {
            display: block;
            animation: slideDown 0.3s ease-out;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .bulk-checkbox {
            transform: scale(1.2);
            margin-right: 8px;
        }

        .item-row.checked {
            background-color: rgba(147, 51, 234, 0.1);
            border-left: 4px solid #9333ea;
        }

        .bulk-action-button {
            transition: all 0.3s ease;
        }

        .bulk-action-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.2);
        }

        .item-unavailable {
            opacity: 0.5;
            background-color: #f3f4f6;
        }

        .item-unavailable .bulk-checkbox {
            opacity: 0.3;
            cursor: not-allowed;
        }
    </style>
    
    <!-- Notification Container -->
    <div id="notification-container" class="fixed top-20 right-5 z-50" style="display: none;">
        <div id="notification" class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded shadow-md">
            <span id="notification-message"></span>
        </div>
    </div>

    <!-- UPDATED: Add reservation highlight notification -->
    <?php if (!empty($show_reservation)): ?>
    <div id="reservation-highlight-notification" class="fixed top-20 left-1/2 transform -translate-x-1/2 z-40 bg-yellow-100 border border-yellow-400 text-yellow-800 px-6 py-3 rounded-lg shadow-lg">
        <div class="flex items-center">
            <i class="fas fa-search mr-2"></i>
            <span>Menampilkan detail untuk reservasi: <strong><?php echo htmlspecialchars($show_reservation); ?></strong></span>
            <button onclick="closeReservationNotification()" class="ml-4 text-yellow-600 hover:text-yellow-800">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>
    <?php endif; ?>

    <!-- Page Title -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <div class="flex items-center mb-4">
            <i class="fas fa-laptop-code text-purple-600 text-2xl mr-3"></i>
            <h2 class="text-2xl font-bold text-gray-800">Persetujuan Formulir Peminjaman - IT Support</h2>
        </div>
        
        <!-- Tab Navigation -->
        <div class="flex space-x-2 mb-6">
            <button id="tab-process" class="tab-button px-6 py-2 rounded-lg font-medium <?php echo $tab === 'process' ? 'active' : ''; ?>" onclick="switchTab('process')">
                <i class="fas fa-clock mr-2"></i>Transaksi Proses
            </button>
            <button id="tab-completed" class="tab-button px-6 py-2 rounded-lg font-medium <?php echo $tab === 'completed' ? 'active' : ''; ?>" onclick="switchTab('completed')">
                <i class="fas fa-check-circle mr-2"></i>Transaksi Selesai
            </button>
        </div>
        
        <!-- Filters -->
        <form id="filter-form" method="GET" class="mb-6">
            <input type="hidden" name="tab" id="current-tab" value="<?php echo $tab; ?>">
            <?php if (!empty($show_reservation)): ?>
            <input type="hidden" name="show_reservation" value="<?php echo htmlspecialchars($show_reservation); ?>">
            <?php endif; ?>
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <div>
                    <label for="borrower" class="block text-sm font-medium text-gray-700 mb-1">Peminjam</label>
                    <select name="borrower" id="borrower" class="w-full p-2 border rounded-md">
                        <option value="">Semua Peminjam</option>
                        <?php 
                        $borrowers_result->data_seek(0); // Reset pointer
                        while ($borrower = $borrowers_result->fetch_assoc()): ?>
                            <option value="<?php echo $borrower['id']; ?>" <?php echo ($filter_borrower == $borrower['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($borrower['username'] . ' (' . $borrower['lokasi_posisi'] . ')'); ?>
                            </option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <div id="status-filter" style="<?php echo $tab === 'completed' ? 'display: none;' : ''; ?>">
                    <label for="status" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select name="status" id="status" class="w-full p-2 border rounded-md">
                        <option value="">Semua Status</option>
                        <option value="Diajukan" <?php echo ($filter_status == 'Diajukan') ? 'selected' : ''; ?>>Diajukan</option>
                        <option value="Disetujui" <?php echo ($filter_status == 'Disetujui') ? 'selected' : ''; ?>>Disetujui</option>
                        <option value="Ditolak" <?php echo ($filter_status == 'Ditolak') ? 'selected' : ''; ?>>Ditolak</option>
                        <option value="OnProccesing" <?php echo ($filter_status == 'OnProccesing') ? 'selected' : ''; ?>>OnProcessing</option>
                        <option value="Dipinjam" <?php echo ($filter_status == 'Dipinjam') ? 'selected' : ''; ?>>Dipinjam</option>
                    </select>
                </div>
                
                <div>
                    <label for="date_start" class="block text-sm font-medium text-gray-700 mb-1">Tanggal Mulai</label>
                    <input type="date" name="date_start" id="date_start" value="<?php echo $filter_date_start; ?>" class="w-full p-2 border rounded-md">
                </div>
                
                <div>
                    <label for="date_end" class="block text-sm font-medium text-gray-700 mb-1">Tanggal Akhir</label>
                    <input type="date" name="date_end" id="date_end" value="<?php echo $filter_date_end; ?>" class="w-full p-2 border rounded-md">
                </div>
            </div>
            
            <div class="mt-4 flex justify-end">
                <button type="submit" class="px-4 py-2 bg-purple-500 text-white rounded-md hover:bg-purple-600">
                    <i class="fas fa-filter mr-2"></i> Filter
                </button>
                <button type="button" onclick="resetFilters()" class="ml-2 px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600">
                    <i class="fas fa-sync mr-2"></i> Reset
                </button>
            </div>
        </form>
    </div>
    
    <!-- Transactions Table -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <h3 class="text-xl font-semibold text-gray-800 mb-4">
            <i class="fas fa-list mr-2 text-purple-600"></i>
            <?php echo $tab === 'process' ? 'Daftar Transaksi Proses' : 'Daftar Transaksi Selesai'; ?>
        </h3>
        
        <div class="overflow-x-auto">
            <table class="min-w-full bg-white">
                <thead class="bg-purple-50">
                    <tr>
                        <th class="py-3 px-4 text-left" style="display: none;">ID</th>
                        <th class="py-3 px-4 text-left">No. Reservasi</th>
                        <th class="py-3 px-4 text-left">Peminjam</th>
                        <th class="py-3 px-4 text-left">Tanggal Transaksi</th>
                        <th class="py-3 px-4 text-left">Tanggal Pengembalian</th>
                        <?php if ($tab === 'completed'): ?>
                        <th class="py-3 px-4 text-left">Status Waktu</th>
                        <?php endif; ?>
                        <th class="py-3 px-4 text-left">Status</th>
                        <th class="py-3 px-4 text-left">Status Peminjaman</th>
                        <th class="py-3 px-4 text-left">Aksi</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($transactions->num_rows > 0): ?>
                        <?php while ($transaction = $transactions->fetch_assoc()): ?>
                            <tr class="border-b hover:bg-purple-50 <?php echo (!empty($show_reservation) && $transaction['reservation_number'] === $show_reservation) ? 'target-reservation' : ''; ?>" 
                                data-transaction-id="<?php echo $transaction['id']; ?>" 
                                data-reservation="<?php echo htmlspecialchars($transaction['reservation_number']); ?>">
                                <td class="py-3 px-4" style="display: none;"><?php echo $transaction['id']; ?></td>
                                <td class="py-3 px-4 <?php echo (!empty($show_reservation) && $transaction['reservation_number'] === $show_reservation) ? 'font-bold text-purple-600' : ''; ?>">
                                    <?php echo htmlspecialchars($transaction['reservation_number']); ?>
                                </td>
                                <td class="py-3 px-4"><?php echo htmlspecialchars($transaction['borrower_name']); ?> (<?php echo htmlspecialchars($transaction['lokasi_posisi']); ?>)</td>
                                <td class="py-3 px-4"><?php echo date('d M Y H:i', strtotime($transaction['tanggal_transaksi'])); ?></td>
                                <td class="py-3 px-4"><?php echo date('d M Y H:i', strtotime($transaction['tanggal_pengembalian'])); ?></td>
                                <?php if ($tab === 'completed'): ?>
                                <td class="py-3 px-4">
                                    <?php 
                                    $status_waktu_class = '';
                                    $status_waktu_text = '-';
                                    
                                    // Check if we have a status_waktu from any of the transaction items
                                    $status_query = "SELECT status_waktu FROM transaction_items WHERE transaction_id = ? AND dikembalikan = 1 LIMIT 1";
                                    $status_stmt = $conn->prepare($status_query);
                                    $status_stmt->bind_param('i', $transaction['id']);
                                    $status_stmt->execute();
                                    $status_result = $status_stmt->get_result();
                                    
                                    if ($status_result->num_rows > 0) {
                                        $status_row = $status_result->fetch_assoc();
                                        switch ($status_row['status_waktu']) {
                                            case 'tepat':
                                                $status_waktu_text = 'Tepat Waktu';
                                                $status_waktu_class = 'bg-green-100 text-green-800';
                                                break;
                                            case 'overdue':
                                                $status_waktu_text = 'Terlambat';
                                                $status_waktu_class = 'bg-red-100 text-red-800';
                                                break;
                                            default:
                                                $status_waktu_text = '-';
                                                $status_waktu_class = 'bg-gray-100 text-gray-800';
                                        }
                                    }
                                    $status_stmt->close();
                                    ?>
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold <?php echo $status_waktu_class; ?>">
                                        <?php echo $status_waktu_text; ?>
                                    </span>
                                </td>
                                <?php endif; ?>
                                <td class="py-3 px-4">
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold
                                        <?php 
                                        switch ($transaction['status']) {
                                            case 'Diajukan':
                                                echo 'bg-yellow-100 text-yellow-800';
                                                break;
                                            case 'Disetujui':
                                                echo 'bg-green-100 text-green-800';
                                                break;
                                            case 'Ditolak':
                                                echo 'bg-red-100 text-red-800';
                                                break;
                                            case 'OnProccesing':
                                                echo 'bg-blue-100 text-blue-800';
                                                break;
                                            default:
                                                echo 'bg-gray-100 text-gray-800';
                                        }
                                        ?>">
                                        <?php echo $transaction['status']; ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4">
                                    <span class="px-2 py-1 rounded-full text-xs font-semibold
                                        <?php 
                                        switch ($transaction['status_peminjaman']) {
                                            case 'None':
                                                echo 'bg-gray-100 text-gray-800';
                                                break;
                                            case 'Dipinjam':
                                                echo 'bg-blue-100 text-blue-800';
                                                break;
                                            case 'Dikembalikan':
                                                echo 'bg-green-100 text-green-800';
                                                break;
                                            default:
                                                echo 'bg-gray-100 text-gray-800';
                                        }
                                        ?>">
                                        <?php echo $transaction['status_peminjaman']; ?>
                                    </span>
                                </td>
                                <td class="py-3 px-4">
                                    <button type="button" class="px-3 py-1 bg-purple-500 text-white rounded hover:bg-purple-600 view-details" 
                                            data-transaction-id="<?php echo $transaction['id']; ?>">
                                        <i class="fas fa-eye mr-1"></i> Detail
                                    </button>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="<?php echo $tab === 'completed' ? '9' : '8'; ?>" class="py-4 px-4 text-center text-gray-500">
                                <?php echo $tab === 'process' ? 'Tidak ada transaksi proses ditemukan' : 'Tidak ada transaksi selesai ditemukan'; ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Transaction Details Modal -->
    <div id="details-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg shadow-xl w-11/12 md:w-3/4 lg:w-2/3 max-h-[90vh] overflow-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <div class="flex items-center space-x-4">
                        <h3 class="text-xl font-semibold text-gray-800">
                            <i class="fas fa-laptop-code text-purple-600 mr-2"></i>
                            Detail Peminjaman - IT Support
                        </h3>
                        <span id="reservation-number-display" class="px-3 py-1 bg-purple-100 text-purple-800 rounded-full text-sm font-medium"></span>
                    </div>
                    <button type="button" class="text-gray-500 hover:text-gray-700 close-modal">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <div id="transaction-details" class="mb-6">
                    <!-- Transaction details will be loaded here -->
                </div>

                <!-- NEW: Bulk Action Bar -->
                <div id="bulk-action-bar" class="bulk-action-bar">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center text-white">
                            <input type="checkbox" id="select-all-items" class="bulk-checkbox mr-3">
                            <label for="select-all-items" class="font-medium">
                                <i class="fas fa-check-double mr-2"></i>
                                Pilih Semua Item Tersedia
                            </label>
                            <span id="selected-count" class="ml-4 px-3 py-1 bg-white bg-opacity-20 rounded-full text-sm">
                                0 item dipilih
                            </span>
                        </div>
                        <div class="flex space-x-2">
                            <button type="button" class="bulk-action-button px-4 py-2 bg-green-500 text-white rounded-lg hover:bg-green-600"
                                    data-bulk-action="approve">
                                <i class="fas fa-check mr-2"></i>Setujui Terpilih
                            </button>
                            <button type="button" class="bulk-action-button px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600"
                                    data-bulk-action="reject">
                                <i class="fas fa-times mr-2"></i>Tolak Terpilih
                            </button>
                            <button type="button" class="bulk-action-button px-4 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600"
                                    data-bulk-action="onprocess">
                                <i class="fas fa-clock mr-2"></i>OnProcess Terpilih
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="border-t pt-4">
                    <h4 class="text-lg font-medium text-gray-700 mb-3">
                        <i class="fas fa-list-check mr-2 text-purple-600"></i>
                        Daftar Item yang Dipinjam
                    </h4>
                    <div id="transaction-items" class="space-y-4">
                        <!-- Transaction items will be loaded here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Approval/Rejection Modal -->
    <div id="action-modal" class="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center hidden">
        <div class="bg-white rounded-lg shadow-xl w-11/12 md:w-2/3 lg:w-1/2 max-h-[90vh] overflow-auto">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 id="action-modal-title" class="text-xl font-semibold text-gray-800">
                        <i class="fas fa-laptop-code text-purple-600 mr-2"></i>
                        Aksi Peminjaman - IT Support
                    </h3>
                    <button type="button" class="text-gray-500 hover:text-gray-700 close-action-modal">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <form id="action-form" class="space-y-4">
                    <input type="hidden" id="action-item-id" name="item_id">
                    <input type="hidden" id="action-transaction-id" name="transaction_id">
                    <input type="hidden" id="action-type" name="action">
                    <input type="hidden" id="bulk-action-flag" name="bulk_action" value="false">
                    <input type="hidden" id="bulk-item-ids" name="item_ids">
                    
                    <div id="note-container" class="hidden">
                        <label for="action-note" class="block text-sm font-medium text-gray-700 mb-1">Catatan</label>
                        <textarea id="action-note" name="note" rows="3" class="w-full p-2 border rounded-md"></textarea>
                    </div>
                    
                    <div id="condition-container" class="hidden">
                        <label class="block text-sm font-medium text-gray-700 mb-1">Kondisi Pengembalian</label>
                        <div class="flex space-x-4">
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Baik" class="form-radio" checked>
                                <span class="ml-2">Baik</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Rusak" class="form-radio">
                                <span class="ml-2">Rusak</span>
                            </label>
                            <label class="inline-flex items-center">
                                <input type="radio" name="condition" value="Hilang" class="form-radio">
                                <span class="ml-2">Hilang</span>
                            </label>
                        </div>
                    </div>
                    
                    <!-- NEW: Bulk action summary -->
                    <div id="bulk-summary" class="hidden">
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                            <h5 class="font-medium text-blue-800 mb-2">
                                <i class="fas fa-info-circle mr-2"></i>Ringkasan Aksi Massal
                            </h5>
                            <p id="bulk-summary-text" class="text-blue-700 text-sm"></p>
                        </div>
                    </div>
                    
                    <div class="flex justify-end pt-4">
                        <button type="button" class="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600 mr-2 close-action-modal">
                            Batal
                        </button>
                        <button type="submit" id="submit-action" class="px-4 py-2 bg-purple-500 text-white rounded-md hover:bg-purple-600">
                            Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</main>

<!-- UPDATED: Add auto-show variables for JavaScript -->
<script>
// Auto-show variables
const AUTO_SHOW_TRANSACTION_ID = <?php echo $auto_show_transaction_id ? $auto_show_transaction_id : 'null'; ?>;
const SHOW_RESERVATION = '<?php echo htmlspecialchars($show_reservation); ?>';

// UPDATED: Function to close reservation notification
function closeReservationNotification() {
    const notification = document.getElementById('reservation-highlight-notification');
    if (notification) {
        notification.style.display = 'none';
    }
    // Remove highlight from target reservation
    const targetRow = document.querySelector('.target-reservation');
    if (targetRow) {
        targetRow.classList.remove('target-reservation', 'highlight-reservation');
    }
}

// UPDATED: Add auto-show functionality when page loads
document.addEventListener('DOMContentLoaded', function() {
    // Highlight target reservation if exists
    if (SHOW_RESERVATION) {
        const targetRow = document.querySelector('.target-reservation');
        if (targetRow) {
            targetRow.classList.add('highlight-reservation');
            // Scroll to the row
            targetRow.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    }
    
    // Auto-show modal if transaction ID is provided
    if (AUTO_SHOW_TRANSACTION_ID) {
        setTimeout(function() {
            // Find and click the detail button for the target transaction
            const detailButton = document.querySelector(`[data-transaction-id="${AUTO_SHOW_TRANSACTION_ID}"] .view-details`);
            if (detailButton) {
                detailButton.click();
            }
        }, 1000); // Delay to allow page to fully load
    }
    
    // Auto-hide reservation notification after 10 seconds
    if (SHOW_RESERVATION) {
        setTimeout(function() {
            closeReservationNotification();
        }, 10000);
    }
});
</script>

<!-- Include JavaScript functions -->
<?php include 'transaction_ui_it.php'; ?>

<?php include '../template_it/footer.php'; ?>