<?php
require_once "config/connection.php";

// Get all buyers for filter
$buyers_query = "SELECT * FROM buyers ORDER BY nama_buyer";
$buyers_result = mysqli_query($conn, $buyers_query);

// Get all color components for filter
$components_query = "SELECT * FROM color_components ORDER BY component_name";
$components_result = mysqli_query($conn, $components_query);

// Get status counts for cards
$ongoing_query = "SELECT COUNT(DISTINCT st.id) as count 
                  FROM stock_transactions st
                  JOIN transaction_items ti ON st.id = ti.transaction_id
                  WHERE st.status_peminjaman = 'Dipinjam'";
$ongoing_result = mysqli_query($conn, $ongoing_query);
$ongoing_count = mysqli_fetch_assoc($ongoing_result)['count'];

$borrowed_query = "SELECT COUNT(ti.id) as count 
                   FROM transaction_items ti
                   JOIN stock_transactions st ON ti.transaction_id = st.id
                   WHERE ti.dikembalikan = 0 AND st.status_peminjaman = 'Dipinjam'";
$borrowed_result = mysqli_query($conn, $borrowed_query);
$borrowed_count = mysqli_fetch_assoc($borrowed_result)['count'];

$overdue_query = "SELECT COUNT(DISTINCT st.id) as count 
                  FROM stock_transactions st
                  JOIN transaction_items ti ON st.id = ti.transaction_id
                  WHERE st.status_peminjaman = 'Dipinjam' 
                  AND ti.status_waktu = 'overdue'";
$overdue_result = mysqli_query($conn, $overdue_query);
$overdue_count = mysqli_fetch_assoc($overdue_result)['count'];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Color Access System Monitor</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8">
        <h1 class="text-3xl font-bold text-center mb-8">Color Access System Monitor</h1>
        
        <!-- Cards Section -->
        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
            <!-- Total On Going -->
            <div class="bg-white rounded-lg shadow p-6 cursor-pointer" onclick="showOngoingModal()">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500">Total On Going</p>
                        <h2 class="text-2xl font-bold"><?php echo $ongoing_count; ?></h2>
                    </div>
                    <div class="bg-blue-100 p-3 rounded-full">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-blue-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                        </svg>
                    </div>
                </div>
            </div>
            
            <!-- Total Dipinjam -->
            <div class="bg-white rounded-lg shadow p-6 cursor-pointer" onclick="showBorrowedModal()">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500">Total Dipinjam</p>
                        <h2 class="text-2xl font-bold"><?php echo $borrowed_count; ?></h2>
                    </div>
                    <div class="bg-green-100 p-3 rounded-full">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-green-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                        </svg>
                    </div>
                </div>
            </div>
            
            <!-- Total Overdue -->
            <div class="bg-white rounded-lg shadow p-6 cursor-pointer" onclick="showOverdueModal()">
                <div class="flex items-center justify-between">
                    <div>
                        <p class="text-gray-500">Total Overdue</p>
                        <h2 class="text-2xl font-bold"><?php echo $overdue_count; ?></h2>
                    </div>
                    <div class="bg-red-100 p-3 rounded-full">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-6 w-6 text-red-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                        </svg>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Filters Section -->
        <div class="bg-white rounded-lg shadow p-6 mb-8">
            <h2 class="text-xl font-semibold mb-4">Filters</h2>
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                <!-- Buyer Filter -->
                <div>
                    <label for="buyer_filter" class="block text-sm font-medium text-gray-700 mb-1">Buyer</label>
                    <select id="buyer_filter" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        <option value="">All Buyers</option>
                        <?php while($buyer = mysqli_fetch_assoc($buyers_result)): ?>
                            <option value="<?php echo $buyer['id']; ?>"><?php echo $buyer['nama_buyer']; ?></option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <!-- Color Code Filter -->
                <div>
                    <label for="color_filter" class="block text-sm font-medium text-gray-700 mb-1">Color Code</label>
                    <select id="color_filter" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" disabled>
                        <option value="">Select Buyer First</option>
                    </select>
                </div>
                
                <!-- Component Filter -->
                <div>
                    <label for="component_filter" class="block text-sm font-medium text-gray-700 mb-1">Component</label>
                    <select id="component_filter" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        <option value="">All Components</option>
                        <?php while($component = mysqli_fetch_assoc($components_result)): ?>
                            <option value="<?php echo $component['id']; ?>"><?php echo $component['component_name']; ?></option>
                        <?php endwhile; ?>
                    </select>
                </div>
                
                <!-- Status Filter -->
                <div>
                    <label for="status_filter" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select id="status_filter" class="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        <option value="">All Status</option>
                        <option value="Tersedia">Tersedia</option>
                        <option value="Dipinjam">Dipinjam</option>
                        <option value="Rusak">Rusak</option>
                        <option value="Hilang">Hilang</option>
                        <option value="Expired">Expired</option>
                        <option value="Renew">Renew</option>
                    </select>
                </div>
            </div>
        </div>
        
        <!-- Search Section -->
        <div class="bg-white rounded-lg shadow p-6 mb-8">
            <h2 class="text-xl font-semibold mb-4">Search</h2>
            <div class="relative">
                <input type="text" id="search_input" class="w-full px-4 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500" placeholder="Search by Code Barang or Color Code...">
                <div class="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                    </svg>
                </div>
            </div>
        </div>
        
        <!-- Color Panels Section -->
        <div id="color_panels_container" class="space-y-6">
            <!-- Color panels will be loaded here via AJAX -->
        </div>
    </div>
    
    <!-- Modals will be added here -->
    <div id="ongoing_modal" class="fixed inset-0 z-50 hidden overflow-y-auto">
        <!-- Modal content -->
    </div>
    
    <div id="borrowed_modal" class="fixed inset-0 z-50 hidden overflow-y-auto">
        <!-- Modal content -->
    </div>
    
    <div id="overdue_modal" class="fixed inset-0 z-50 hidden overflow-y-auto">
        <!-- Modal content -->
    </div>
    
    <div id="item_detail_modal" class="fixed inset-0 z-50 hidden overflow-y-auto">
        <!-- Modal content -->
    </div>
    
    <script>
    $(document).ready(function() {
        // Load initial data
        loadColorPanels();
        
        // Filter change events
        $('#buyer_filter, #color_filter, #component_filter, #status_filter').change(function() {
            loadColorPanels();
        });
        
        // Search input event
        $('#search_input').on('input', function() {
            const searchTerm = $(this).val().trim();
            if (searchTerm.length > 0) {
                $('.color-panel, .stock-item').removeClass('bg-yellow-100');
                
                // Search by code barang
                $('.stock-item').each(function() {
                    const codeBarang = $(this).data('code-barang');
                    if (codeBarang.includes(searchTerm)) {
                        $(this).addClass('bg-yellow-100');
                        $(this).closest('.color-panel').show();
                        $(this).show();
                    } else {
                        $(this).hide();
                    }
                });
                
                // If no code barang matches, search by color code
                if ($('.stock-item.bg-yellow-100').length === 0) {
                    $('.color-panel').each(function() {
                        const colorCode = $(this).data('color-code');
                        if (colorCode.includes(searchTerm)) {
                            $(this).addClass('bg-yellow-100').show();
                            $(this).find('.stock-item').show();
                        } else {
                            $(this).hide();
                        }
                    });
                }
            } else {
                $('.color-panel, .stock-item').show().removeClass('bg-yellow-100');
            }
        });
        
        // Get colors when buyer is selected
        $('#buyer_filter').change(function() {
            const buyerId = $(this).val();
            if (buyerId) {
                $.ajax({
                    url: 'get_colors.php',
                    type: 'POST',
                    data: { buyer_id: buyerId },
                    success: function(response) {
                        $('#color_filter').html(response).prop('disabled', false);
                        loadColorPanels();
                    }
                });
            } else {
                $('#color_filter').html('<option value="">Select Buyer First</option>').prop('disabled', true);
                loadColorPanels();
            }
        });
    });
    
    function loadColorPanels() {
        const buyerId = $('#buyer_filter').val();
        const colorId = $('#color_filter').val();
        const componentId = $('#component_filter').val();
        const status = $('#status_filter').val();
        
        $.ajax({
            url: 'get_color_panels.php',
            type: 'POST',
            data: {
                buyer_id: buyerId,
                color_id: colorId,
                component_id: componentId,
                status: status
            },
            success: function(response) {
                $('#color_panels_container').html(response);
                
                // Add click events for color panels
                $('.color-panel-header').click(function() {
                    $(this).next('.stock-items-container').slideToggle();
                });
                
                // Add click events for borrowed items
                $('.stock-item[data-status="Dipinjam"]').click(function() {
                    const itemId = $(this).data('item-id');
                    showItemDetailModal(itemId);
                });
            }
        });
    }
    
    function showOngoingModal() {
        $.ajax({
            url: 'get_ongoing_transactions.php',
            type: 'GET',
            success: function(response) {
                $('#ongoing_modal').html(response).removeClass('hidden');
            }
        });
    }
    
    function showBorrowedModal() {
        $.ajax({
            url: 'get_borrowed_items.php',
            type: 'GET',
            success: function(response) {
                $('#borrowed_modal').html(response).removeClass('hidden');
            }
        });
    }
    
    function showOverdueModal() {
        $.ajax({
            url: 'get_overdue_transactions.php',
            type: 'GET',
            success: function(response) {
                $('#overdue_modal').html(response).removeClass('hidden');
            }
        });
    }
    
    function showItemDetailModal(itemId) {
        $.ajax({
            url: 'get_item_detail.php',
            type: 'POST',
            data: { item_id: itemId },
            success: function(response) {
                $('#item_detail_modal').html(response).removeClass('hidden');
            }
        });
    }
    
    function closeModal(modalId) {
        $('#' + modalId).addClass('hidden');
    }
    </script>
</body>
</html>